<?php
/**
 * Created by PhpStorm.
 * User: user
 * Date: 27.03.17

 */

namespace App\Classes\Parsing;

use App\Mail\TemplateEmail;
use App\Models\Aircraft;
use App\Models\AircraftType;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\CALMessage;
use App\Models\Country;
use App\Models\CrewListMessage;
use App\Models\EmailNotification;
use App\Models\Flight;
use App\Models\FlightDelay;
use App\Models\FlightMessage;
use App\Models\FlightNumber;
use App\Models\HBGMessage;
use App\Models\LIRMessage;
use App\Models\Passenger;
use App\Models\PassengerFlight;
use App\Models\PassengerFlightAct;
use App\Models\PassengerFrequentFlyer;
use App\Models\PassengerOrder;

use App\Models\CPMMessage;
use App\Models\LDMmessage;
use App\Models\LPMMessage;
use App\Models\MVTMessage;
use App\Models\OtherMessage;
use App\Models\PNLMessage;
use App\Models\PRLMessage;
use App\Models\PSMMessage;
use App\Models\PTMMessage;
use App\Models\SLSMessage;
use App\Models\TPMMessage;
use App\Models\UCMMessage;
use Doctrine\DBAL\Driver\Mysqli\MysqliException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class Parse
{
    const DEL = "DEL";
    const CHG = "CHG";
    const ADD = "ADD";

    public static $months = [
        "JAN" => "01",
        "FEB" => "02",
        "MAR" => "03",
        "APR" => "04",
        "MAY" => "05",
        "JUN" => "06",
        "JUL" => "07",
        "AUG" => "08",
        "SEP" => "09",
        "OCT" => "10",
        "NOV" => "11",
        "DEC" => "12"
    ];



    public static function checkAndUpdateAirport(&$messageDetails){
        $exceptionAPList = ["RBD"];

        if (!$messageDetails || !count($messageDetails)){
            return;
        }

        if (isset($messageDetails["departureAirport"]) && $messageDetails["departureAirport"]){
            if (in_array($messageDetails["departureAirport"], $exceptionAPList)){
                $messageDetails["departureAirport"] = null;
            }
        }

        if (isset($messageDetails["arrivalAirport"]) && $messageDetails["arrivalAirport"]){
            if (in_array($messageDetails["arrivalAirport"], $exceptionAPList)){
                $messageDetails["arrivalAirport"] = null;
            }
        }
    }


    public static function sentDateCheck($givenDay, $sentDate = null, ParseEmail $parseEmail = null, $skipDateRangeCheck = null, $departureMonth = null){

        $twoDaysInSecs = 2*24*60*60;

        if ($skipDateRangeCheck){
            $twoDaysInSecs = 7*24*60*60;
        }

        $sentDate = $sentDate ? $sentDate : date("Y-m-d");

        if (!$givenDay){
            $givenDay = date("d");
        }

        // If Month and Day are mentioned take it
        if ($departureMonth && array_key_exists($departureMonth, self::$months)){

            $month = self::$months[$departureMonth];

            $date = validateMVTDate(date("Y-".$month."-".$givenDay, strtotime($sentDate)));

//            debug("RESULT: MONTH AND DAY MENTIONED! ORIGINAL DATE = ".$date);

            return $date;
        }
        else {
            $date = validateMVTDate(date("Y-m-".$givenDay, strtotime($sentDate)));
        }

        debug("SENT DATE = ".$sentDate);
        debug("ORIG DATE INIT = ".$date);

        $curDate = date("Y-m-d");

        // Problematic SKIP
        if (strtotime($date) > strtotime($curDate) + $twoDaysInSecs){
//        if (strtotime($date) > strtotime($sentDate) + $twoDaysInSecs){

            $days = getMonthDays($date);
            $prevMonthDate = date("Y-m-{$givenDay}",  strtotime("-{$days} days", strtotime($date)));

            if ((strtotime($prevMonthDate) <= strtotime($sentDate) + $twoDaysInSecs) &&
                (strtotime($prevMonthDate) >= strtotime($sentDate) - 3*24*60*60)){

                debug("RESULT: PREV MONTH DATE = ".$prevMonthDate);

                // Set date
                return $prevMonthDate;
            }

            debug("ERROR: strtotime($date) > strtotime($curDate) + $twoDaysInSecs");
            return false;

        }
        else {
            if ($skipDateRangeCheck){
                $monthDif =  30*24*60*60;
                if (strtotime($date) < strtotime($curDate) -  $monthDif){
                    debug("RESULT: ERROR! 1 MONTH RANGE EXCEEDED! ORIGINAL DATE = ".$date);
                    return false;
                }
            }
            // Problematic SKIP
            else {
                if (strtotime($date) < strtotime($curDate) - $twoDaysInSecs){
                    debug("RESULT: ERROR! 2 DAYS RANGE EXCEEDED! ORIGINAL DATE = ".$date);
                    return false;
                }
            }
        }

//        debug("RESULT: SUCCESS! ORIGINAL DATE = ".$date);
        return $date;
    }

    public static function createMessage($parseText, $messageType, $flight = null, $flightNumber = null, $messageDetails = null, $mvtType = null, $mvtDiversion = null, $skipped = null)
    {
        $messageLog = [];

        $message = null;
//        try {
            // UCM|PSM|PTM|CPM|LPM|TPM|COM|COMRO|WAB
            switch ($messageType) {
                case PNL:
                case ADL:
                    $message = new PNLMessage();
                    $message->type = $messageType;
                    $message->part = isset($messageDetails['part']) && $messageDetails['part'] ? $messageDetails['part'] : null;
                    $message->pax_y = isset($messageDetails['paxY']) && $messageDetails['paxY'] ? $messageDetails['paxY'] : null;
                    break;

                case PRL:
                    $message = new PRLMessage();
                    $message->part = isset($messageDetails['part']) && $messageDetails['part'] ? $messageDetails['part'] : null;
                    $message->pax_y = isset($messageDetails['paxY']) && $messageDetails['paxY'] ? $messageDetails['paxY'] : null;
                    break;

                case MVT:
                    $message = new MVTMessage();
                    $message->type = $mvtType;
                    $message->diversion = $mvtDiversion;
                    // MVT Option only
                    $message->skipped = $skipped ? $skipped : null;
                    break;

                case LDM:
                    $message = new LDMmessage();
                    break;

                case UCM:
                    $message = new UCMMessage();
                    break;

                case PSM:
                    $message = new PSMMessage();
                    break;

                case PTM:
                    $message = new PTMMessage();
                    break;

                case CPM:
                    $message = new CPMMessage();
                    break;

                case LPM:
                    $message = new LPMMessage();
                    break;

                case TPM:
                    $message = new TPMMessage();
                    break;

                case SLS:
                    $message = new SLSMessage();
                    break;

                case HBG:
                    $message = new HBGMessage();
                    break;

                case LIR:
                    $message = new LIRMessage();
                    break;

                case CAL:
                case PAL:
                    $message = new CALMessage();
                    break;

                case CREW_LIST:
                    $message = new CrewListMessage();
                    break;

                case NOT:
                case NOTOC:
                case WAB:
                case COM:
                case COMRO:
                case TXT_ING:
                case ONLOAD:
                case LDS:
                case LOADSHEET:
                case LOADSHEET_FINAL:
                    $message = new OtherMessage();
                    $message->type = $messageType;
                    break;

                default:
                    $message = new OtherMessage();
                    $message->type = OTHS;
                    break;
            }
            if ($flight) {
                $message->flight_id = $flight->id;
                $message->parsed = true;
            }
            if (!$messageDetails) {
                $messageLog[] = "Email body does not match with regExp";
            }


            $message->log = count($messageLog) ? serialize($messageLog) : null;
            $message->flight_number_id = $flightNumber ? $flightNumber->id : null;
            $message->received_datetime = date("Y-m-d H:i:s");
            $message->data = $messageDetails ? serialize($messageDetails) : null;
            $message->email_body = $parseText; //strlen($parseText) > 5000 ? str_limit($parseText, 5000) : $parseText;

            $message->created_by = Auth::user()->id;
            $message->created_at = date("Y-m-d H:i:s");
            $message->updated_at = date("Y-m-d H:i:s");
            $message->save();
//        }
//        catch(\Exception $e){
//            Parse::sendEmailNotification($messageLog, $e->getMessage());
//        }

        if ($flight){
            Parse::createFlightMessage($flight, $messageType, $message);
        }

        return $message;

    }

    public static function updateMessage(&$message, $messageType, $flight = null, $flightNumber = null, $messageDetails = null)
    {
        try {
            switch ($messageType) {
                case PNL:
                case ADL:
                case PRL:
                    $message->part = isset($messageDetails['part']) && $messageDetails['part'] ? $messageDetails['part'] : null;
                    $message->pax_y = isset($messageDetails['paxY']) && $messageDetails['paxY'] ? $messageDetails['paxY'] : null;
                    break;
            }

            $createFlightMessage = !$message->flight_id;

            if ($flight) {
                $message->flight_id = $flight->id;
                $message->parsed = true;
            }

            $message->seen = true;
            $message->flight_number_id = $flightNumber ? $flightNumber->id : null;
            $message->data = $messageDetails ? serialize($messageDetails) : null;
            $message->updated_at = date("Y-m-d H:i:s");
            $message->save();

            if ($createFlightMessage && $flight) {
                Parse::createFlightMessage($flight, $messageType, $message);
            }
        }
        catch(\Exception $e){
            Parse::sendEmailNotification($e->getCode(), $e->getMessage());
        }
    }

    public static function sendDelayEmailNotification($flight){

        $airline = $flight->flightNumber && $flight->flightNumber->airline ? $flight->flightNumber->airline->iata : null;
        $flightNumber = $flight->flightNumber ? $flight->flightNumber->flight_number : "-";
        $depAirport = $flight->flightNumber ? $flight->flightNumber->departureAirport : null;
        $arrAirport = $flight->flightNumber ? $flight->flightNumber->arrivalAirport : null;
        $sector = ($depAirport ? $depAirport->iata : "")."-".($arrAirport ? $arrAirport->iata : "");

        // Only Departing from handling stations
        if ($depAirport && !in_array($depAirport->id, Airport::getHandlingStations())){
            return;
        }

        $totalDelaySecs = 0;

        $notifiedDelays = $otherDelays = [];
        $flightDelays = FlightDelay::with(['delay', 'delay.airline', 'delay.emailNotification'])
                            ->where("flight_id", $flight->id)
                            ->get();

        $now = date("Y-m-d H:i:s");

        foreach ($flightDelays as $flightDelay) {
            if ($flightDelay->delay){
                // Notification
                if ($flightDelay->delay->emailNotification)
                {
                    if ($flightDelay->notification_date && (strtotime($flightDelay->notification_duration) == strtotime($flightDelay->duration)))
                    {
                        // Skip
                    }
                    else
                    {
                        // Send
                        $notifiedDelays[] = $flightDelay;

                        // Save Log
                        $flightDelay->notification_date = $now;
                        $flightDelay->notification_duration = $flightDelay->duration;
                        $flightDelay->save();
                    }
                }

                if ($flightDelay->duration != EMPTY_TIME ) {
                    $totalDelaySecs += TimeToSeconds($flightDelay->duration, true);
                }

                $otherDelays[] = $flightDelay;
            }
        }

        $dep = getFlightDepartureDate($flight, true);
        if (strtotime($dep) < strtotime(date("Y-m-d"))){
            return;
        }


        $totalDelay = secsToTime($totalDelaySecs);

        foreach ($notifiedDelays as $flightDelay) {

            $delay = $flightDelay->delay;
            $currentDelayTime = $flightDelay->duration;

            $otherDelaysStr = "";
            foreach ($otherDelays as $item) {
                if ($item->id != $flightDelay->id) {
                    $otherDelaysStr .= "<br/>DL " . ($item->delay->airline ? $item->delay->airline->iata . " " : "") . $item->delay->code
                        . " | " . baseTimeFormat($item->duration)
                        . " | " . ($item->delay->alpha_code ? $item->delay->alpha_code . " | " : "") . $item->delay->description;
                }
            }

            $messageContent = "Delay Notification:"
                ."<br/>Flight: ".$airline.$flightNumber." (".$sector.")"
                ."<br/>STD: ".($flight->std && $flight->std != EMPTY_DATETIME ? baseDateFormat($flight->std, true) : "N/A")
                ."<br/>".strtoupper(getFlightDepartureTimeType($flight)) . ": " . baseDateFormat(getFlightDepartureDate($flight), true)
                ."<br/>Delay Code: ".($delay->airline ? $delay->airline->iata." " : ""). $delay->code." | ".($delay->alpha_code ? $delay->alpha_code. " | " : "").$delay->description
                ."<br/>Delay Time: ".baseTimeFormat($currentDelayTime)
                ."<br/>--------------------------";

            if ($otherDelaysStr){
                $messageContent .=  "<br/>Other Delays:"
                                   ."{$otherDelaysStr}"
                                   ."<br/>--------------------------";
            }

            $messageContent .= "<br/>Total Delay Time: ".($totalDelay ? baseTimeFormat($totalDelay) : "N/A");

            $emails = explode(";", $delay->emailNotification->emails);
            // 7. Send Log email to developer

            $finalRecipients = [];
            foreach ($emails as $each) {
                if (trim($each)){
                    $finalRecipients[] = $each;
                }
            }

            debug("DELAY NOTIF");
            debug($finalRecipients);
            debug($messageContent);

            if (count($finalRecipients)){
                Mail::to($finalRecipients)
                    ->bcc("dilovar88@gmail.com")
//                    ->bcc("dilovar88@mail.ru")
                    ->send(new TemplateEmail($messageContent, "Delay Notification"));
            }
        }

    }

    public static function sendAutoCreatedEmailNotification($flight){

        $airline = $flight->flightNumber && $flight->flightNumber->airline ? $flight->flightNumber->airline->iata : null;
        $flightNumber = $flight->flightNumber ? $flight->flightNumber->flight_number : "-";
        $depAirport = $flight->flightNumber ? $flight->flightNumber->departureAirport : null;
        $arrAirport = $flight->flightNumber ? $flight->flightNumber->arrivalAirport : null;
        $sector = ($depAirport ? $depAirport->iata : "")."-".($arrAirport ? $arrAirport->iata : "");

        // Only Departing from handling stations
        if ($depAirport && !in_array($depAirport->id, Airport::getHandlingStations())){
            return;
        }

        $messageContent = "Auto-created Flight Notification:"
            ."<br/>Flight: ".$airline.$flightNumber." (".$sector.")"
            ."<br/>STD: ".($flight->std && $flight->std != EMPTY_DATETIME ? baseDateFormat($flight->std, true) : "N/A")
            ."<br/>".strtoupper(getFlightDepartureTimeType($flight)) . ": " . baseDateFormat(getFlightDepartureDate($flight), true)
            ."<br/>".strtoupper(getFlightArrivalTimeType($flight)) . ": " . baseDateFormat(getFlightArrivalDate($flight), true);

        $emails = EmailNotification::getAutoCreationFlightEmails();
        // 7. Send Log email to developer

        $finalRecipients = [];
        foreach ($emails as $each) {
            if (trim($each)){
                $finalRecipients[] = $each;
            }
        }

        if (count($finalRecipients) && app()->environment(PRODUCTION)){
            Mail::to($finalRecipients)
    //          ->bcc("dilovar88@mail.ru")
                ->send(new TemplateEmail($messageContent, "Delay Notification"));
        }
    }

    public static function sendDiversionEmailNotification($flight){

        $airline = $flight->flightNumber && $flight->flightNumber->airline ? $flight->flightNumber->airline->iata : null;
        $flightNumber = $flight->flightNumber ? $flight->flightNumber->flight_number : "-";
        $depAirport = $flight->flightNumber ? $flight->flightNumber->departureAirport : null;
        $arrAirport = $flight->flightNumber ? $flight->flightNumber->arrivalAirport : null;
        $sector = ($depAirport ? $depAirport->iata : "")."-".($arrAirport ? $arrAirport->iata : "");

        // Only Departing from handling stations
        if ($depAirport && !in_array($depAirport->id, Airport::getHandlingStations())){
            return;
        }

        $messageContent = "Diversion Flight Notification:"
            ."<br/>Flight: ".$airline.$flightNumber." (".$sector.")"
            ."<br/>Diverted Airport"." (".($flight->divertedAirport ? $flight->divertedAirport->iata : "-").")"
            ."<br/>STD: ".($flight->std && $flight->std != EMPTY_DATETIME ? baseDateFormat($flight->std, true) : "N/A")
            ."<br/>".strtoupper(getFlightDepartureTimeType($flight)) . ": " . baseDateFormat(getFlightDepartureDate($flight), true)
            ."<br/>".strtoupper(getFlightArrivalTimeType($flight)) . ": " . baseDateFormat(getFlightArrivalDate($flight), true);

        $emails = EmailNotification::getDiversionFlightEmails();
        // 7. Send Log email to developer

        $finalRecipients = [];
        foreach ($emails as $each) {
            if (trim($each)){
                $finalRecipients[] = $each;
            }
        }

        if (count($finalRecipients)){
            Mail::to($finalRecipients)
//                    ->bcc("dilovar88@mail.ru")
                ->send(new TemplateEmail($messageContent, "Delay Notification"));
        }
    }

    public static function sendEmailNotification($messageLog, $errorMessage)
    {
        if (app()->environment() != PRODUCTION){
            return;
        }

        $totalMessage = array_merge(["LOG-------------"],
            ( $messageLog && is_array($messageLog) ? $messageLog : []),
            ["ERRORS-------------"],
            ($errorMessage && is_array($errorMessage) ? $errorMessage : [$errorMessage])
        );

        // 7. Send Log email to developer
        $recipientEmail = "dilovar88@mail.ru";
        $recipientName = "Dilovar Tursunov";

        Mail::to($recipientEmail)
        //  ->bcc("dilovar88@mail.ru")
            ->send(new TemplateEmail(env("AIRLINE")." - Error Log", print_r($totalMessage, TRUE)));
//            ->send(new TemplateEmail(print_r($totalMessage, TRUE), env("AIRLINE")." - Error Log"));
    }

    public static function sendSimpleEmailNotification($bodyText)
    {
        if (app()->environment() != PRODUCTION){
            return;
        }

        // 7. Send Log email to developer
        $to = "dilovar88@mail.ru";

        Mail::to($to)
            ->send(new TemplateEmail( $bodyText, env("AIRLINE")." - Error Log"));
    }



    public static function sendSimpleEmail($emails, $subject, $messageContent, $from = "fcc@tajikairlines.com"){
        if (app()->environment() != PRODUCTION){
            return;
        }
        // 7. Send Log email to developer
        /*
        foreach ($emails as $recipientEmail)
        {
        */
        Mail::to($emails)
//            ->bcc("dilovar88@mail.ru")
            ->send(new TemplateEmail($messageContent, $subject));
        //}
    }


    public static function forwardMVTToStations($subject, $emailBody, $flightNumber, $hubStations, $hubEmails, $stationEmails){
        if (!$flightNumber){
            return;
        }

        if (!$subject || !trim($subject)){
            $subject = "MVT";
        }

        $recipientEmails = self::getRecipientsEmails($flightNumber, $hubStations, $hubEmails, $stationEmails);

        debug($recipientEmails);
        if (count($recipientEmails)){
            self::sendSimpleEmail($recipientEmails, $subject, $emailBody);
        }
    }

    public static function forwardMVTToAircraft($subject, $emailBody, $aircraft = null){

        if (!$aircraft || $aircraft->stop_emails || !$aircraft->email_recipients){
            return;
        }

        if (!$subject || !trim($subject)){
            $subject = "MVT";
        }

        $recipientEmails = [];
        $emails = explode(";", $aircraft->email_recipients);
        foreach ($emails as $each) {
            if (!$each || !trim($each)){
                continue;
            }

            if (!in_array(trim($each), $recipientEmails)){
                $recipientEmails[] = trim($each);
            }
        }

        if (count($recipientEmails)){
            self::sendSimpleEmail(array_unique($recipientEmails), $subject, $emailBody);
        }
    }

    public static function getRecipientsEmails($flightNumber, $hubStations = null, $hubEmails = null, $stationEmails = null){
        $recipientEmails = [];

        if (! ($hubStations && $hubEmails)){
            $result = self::setHubStationsAndEmails();
            $hubStations = $result[0];
            $hubEmails = $result[1];
        }

        if (!$stationEmails){
            $stationEmails = self::setStationEmailRecipients();
        }

        // Hub stations
        if ($hubEmails && count($hubEmails)){
            foreach ($hubEmails as $each) {
                if (trim($each)){
                    $recipientEmails[] = trim($each);
                }
            }
        }

        if ($stationEmails && count($stationEmails)) {
            // Send to Departure AP if not hub stations
            if ($flightNumber->departure_airport_id) {
                // Not sending emails if it's hub,
                // since hub email stations will receive email at all points
                if ($hubStations && in_array($flightNumber->departure_airport_id, $hubStations)) {
                    // skip
                } else {
                    if (array_key_exists($flightNumber->departure_airport_id, $stationEmails) &&
                        count($stationEmails[$flightNumber->departure_airport_id])
                    ) {
                        foreach ($stationEmails[$flightNumber->departure_airport_id] as $each) {
                            if (trim($each)){
                                $recipientEmails[] = trim($each);
                            }
                        }
                    }
                }
            }

            // Send to Arrival AP if not hub stations
            if ($flightNumber->arrival_airport_id) {
                // Not sending emails if it's hub,
                // since hub email stations will receive email at all points
                if ($hubStations && in_array($flightNumber->arrival_airport_id, $hubStations)) {
                    // skip
                } else {
                    if (array_key_exists($flightNumber->arrival_airport_id, $stationEmails) &&
                        count($stationEmails[$flightNumber->arrival_airport_id])
                    ) {
                        foreach ($stationEmails[$flightNumber->arrival_airport_id] as $each) {
                            if (trim($each)){
                                $recipientEmails[] = trim($each);
                            }
                        }

                    }
                }
            }
        }

        return array_unique($recipientEmails);

    }

    public static function setHubStationsAndEmails(){
        $airports = Airport::where("email_hub", true)
                            ->whereNotNull("email_recipients")
                            ->where("email_recipients", "!=", "")
                            ->whereNull("stop_emails")
                            ->get();

        if (!count($airports)){
            return [null, null];
        }

        $hubEmails = $hubStations = [];
        foreach ($airports as $each) {
            if (!$each->email_recipients || !trim($each->email_recipients)){
                continue;
            }

            // Hub stations
            $hubStations[] = $each->id;

            // Hub emails
            $delimiter = contains(",", $each->email_recipients) ? "," : ";";
            $emails = explode($delimiter, $each->email_recipients);
            foreach ($emails as $email) {
                $hubEmails[] = $email;
            }
        }

        return [
            $hubStations,
            $hubEmails
        ];
    }

    public static function setStationEmailRecipients(){
        $airports = Airport::whereNull("email_hub")
            ->whereNull("stop_emails")
            ->whereNotNull("email_recipients")
            ->where("email_recipients", "!=", "")
            ->get();

        if (!count($airports)){
            return null;
        }

        $stationEmails = [];
        foreach ($airports as $each) {
            if (!$each->email_recipients || !trim($each->email_recipients)){
                continue;
            }

            $delimiter = contains(",", $each->email_recipients) ? "," : ";";
            $emails = explode($delimiter, $each->email_recipients);

            if (!count($emails)){
                continue;
            }

            $stationEmails[$each->id] = [];

            foreach ($emails as $email) {
                if (trim($email)){
                    $stationEmails[$each->id][] = trim($email);
                }
            }
        }

        return $stationEmails;
    }

    public static function createFlightMessage($flight, $messageType, $message){
        if (in_array($messageType, [WAB, COM, COMRO, TXT_ING])){
            $messageType = OTHS;
        }
        if (in_array($messageType, [NOT, NOTOC])){
            $messageType = NOT;
        }

        $msg = new FlightMessage();
        $msg->flight_id = $flight->id;
        $msg->message_type = $messageType;
        $msg->message_id = $message->id;
        $msg->received_datetime = $message->received_datetime;
        $msg->created_by = $message->created_by;
        $msg->save();
    }

    public static function isHandlingAirline($airline = null){
        if (!$airline){
            return false;
        }

        return $airline->handling_airline;
    }


    public static function findFlightProcess($messageDetails, $date, $excludeRequiredAircraft = false, $airportIds = null,
                                             ParseEmail $parseEmail = null, $arrDiversion = false, $mvtType = null){

        $messageLog = [];

        // if Airline code not 2(IATA) or 3(ICAO)
        if (!$messageDetails['airline'] || !in_array(strlen($messageDetails['airline']), [2, 3])){

            $messageLog[] = "Airline Code (".$messageDetails['airline'].") is not correct!";

            if ($parseEmail){
                $parseEmail->messageLog[] = "Airline Code (".$messageDetails['airline'].") is not correct!";
            }

            return [
                "flight"            => null,
                "aircraft"          => null,
                "airline"           => null,
                "flightNumber"      => null,
                "codeShare"         => null,
                "messageLog"        => $messageLog,
            ];
        }

        // Get Airline
        if (strlen($messageDetails["airline"]) == 3){
            $airline = Airline::where("icao", $messageDetails["airline"])->first();
        }
        else {
            $airline = Airline::where("iata", $messageDetails["airline"])->first();
        }

        if (!$airline){

            $messageLog[] = "Airline Not Found: (".$messageDetails['airline'].")!";

            if ($parseEmail){
                $parseEmail->messageLog[] = "Airline Not Found: (".$messageDetails['airline'].")!";
            }

            return [
                "flight"        => null,
                "aircraft"      => null,
                "airline"       => null,
                "flightNumber"  => null,
                "codeShare"     => null,
                "messageLog"    => $messageLog,
            ];
        }

        if (env(CARRIER) == HANDLING && !$airline->handling_airline){
            return [
                "flight"        => null,
                "aircraft"      => null,
                "airline"       => null,
                "flightNumber"  => null,
                "codeShare"     => null,
                "messageLog"    => $messageLog,
            ];
        }


        // Find Aircraft
        $aircraft = null;
        if (isset($messageDetails["aircraftName"])){
            // Find the aircraft
            $aircraft = self::findAircraft($messageDetails["aircraftName"], $airline);
        }

        if (!$excludeRequiredAircraft && !$aircraft)
        {
            $messageLog[] = "Aircraft Not Found: (".$messageDetails['aircraftName'].")!";

            if ($parseEmail){
                $parseEmail->messageLog[] = "Aircraft Not Found: (".$messageDetails['aircraftName'].")!";
            }

        }
        // ---------------

        // Find Flight Number

        // Remove all 0s from beginning
        $messageDetails["flightNumber"] = ltrim($messageDetails["flightNumber"], "0");

        if (!$airportIds){
            $airportIds = Airport::whereNotNull("iata")->pluck("id", "iata")->all();
        }

        // Get flight number record
        $findFlightNumbers = FlightNumber::where("airline_id", $airline->id)
                                         ->where("flight_number", $messageDetails["flightNumber"])
                                         ->whereNull("deleted_at");



        $foundDepartureAirportId = $foundArrivalAirportId = null;
        if (isset($messageDetails["departureAirport"]) && $messageDetails["departureAirport"]){
            if (!isset($airportIds[$messageDetails["departureAirport"]])) {
                $depAirport = self::createAirport($messageDetails["departureAirport"]);
                if ($depAirport){
                    $findFlightNumbers->where("departure_airport_id", $foundDepartureAirportId = $depAirport->id);
                }
            }
            else {
                $findFlightNumbers->where("departure_airport_id", $foundDepartureAirportId = $airportIds[$messageDetails["departureAirport"]]);
            }
        }


        $arrivalSearch = "";
        if (isset($messageDetails["arrivalAirport"]) && $messageDetails["arrivalAirport"]) {
            if (!isset($airportIds[$messageDetails["arrivalAirport"]])) {
                $arrAirport = self::createAirport($messageDetails["arrivalAirport"]);
                $foundArrivalAirportId = $arrAirport->id;
            } else {
                $foundArrivalAirportId = $airportIds[$messageDetails["arrivalAirport"]];
            }

            if ($foundArrivalAirportId){
                if ($arrDiversion){
                    $arrivalSearch = " AND arrival_airport_id = {$foundArrivalAirportId}";
                }
                else {
                    $findFlightNumbers->where("arrival_airport_id", $foundArrivalAirportId);
                }
            }
        }

        $findFlightNumbers = $findFlightNumbers->get();

        $flight = $flightNumber = null;
        $currentDate = date("Y-m-d H:i");
        if ($findFlightNumbers->count() > 0) {
            foreach ($findFlightNumbers as $fnEach) {
                $flights = Flight::where("flight_number_id", "=", $fnEach->id)
//                ->where("aircraft_id", "=", $aircraft->id)
                    ->whereNull("cancelled_at")
                    ->whereNull("deleted_at")
                    ->whereRaw("((s_flights.atd IS NOT NULL AND s_flights.atd != '' AND DATE(s_flights.atd) = '$date') OR
                         (s_flights.etd IS NOT NULL AND s_flights.etd != '' AND DATE(s_flights.etd) = '$date') OR
                         (s_flights.ptd IS NOT NULL AND s_flights.ptd != '' AND DATE(s_flights.ptd) = '$date') OR
                         (s_flights.std IS NOT NULL AND s_flights.std != '' AND DATE(s_flights.std) = '$date'))
                         {$arrivalSearch}")
                    ->get();

                // Iterate if more found
                if (count($flights) > 1) {

                    // Iterate flights
                    foreach ($flights as $flEach) {
                        if (!$mvtType){
                            $flightNumber = $fnEach;
                            $flight = $flEach;
                            break;
                        }
                        else {
                            // DEPARTURE
                            if ($mvtType == MVT_DEPARTURE){
                                $dep = getFlightDepartureDate($flEach);
                                debug("DEPARTURE CASE. FLT DEP: ". $dep. " | NOW: ".$currentDate);
                                if (strtotime($dep) <= strtotime($currentDate)){
                                    $flightNumber = $fnEach;
                                    $flight = $flEach;
                                    break;
                                }
                            }
                            // ARRIVAL
                            else {
                                $arr = getFlightArrivalDate($flEach);
                                debug("ARRIVAL CASE. FLT ARR: ". $arr. " | NOW: ".$currentDate);
                                if (strtotime($arr) <= strtotime($currentDate)){
                                    $flightNumber = $fnEach;
                                    $flight = $flEach;
                                    break;
                                }
                            }
                        }
                    }

                    break;
                }
                // If Flight Found -> Break
                else if (count($flights) == 1){
                    $flightNumber = $fnEach;
                    $flight = $flights->first();
                    debug("MATCHING FLT FOUND: FLT: ".getFlightDepartureDate($flight). " | NOW: ".$currentDate);
                    break;
                }
            }

            // NEW CHECK
            // IF Message is ARRIVAL -> which means that no departure timing is mentioned
            // then we might include arrival date
            if (!$flight && $mvtType && $mvtType == MVT_ARRIVAL){

                foreach ($findFlightNumbers as $fnEach) {
                    $flight = Flight::where("flight_number_id", "=", $fnEach->id)
    //                ->where("aircraft_id", "=", $aircraft->id)
                        ->whereNull("cancelled_at")
                        ->whereNull("deleted_at")
                        ->whereRaw("((s_flights.ata IS NOT NULL AND s_flights.ata != '' AND DATE(s_flights.ata) = '$date') OR
                             (s_flights.eta IS NOT NULL AND s_flights.eta != '' AND DATE(s_flights.eta) = '$date') OR
                             (s_flights.pta IS NOT NULL AND s_flights.pta != '' AND DATE(s_flights.pta) = '$date') OR
                             (s_flights.sta IS NOT NULL AND s_flights.std != '' AND DATE(s_flights.sta) = '$date'))
                             {$arrivalSearch}")
                        ->first();
                        // If Flight Found -> Break
                        if ($flight) {
                            $flightNumber = $fnEach;
                            break;
                        }
                }
            }
        }

        // If Not found matching flight create new Flight number
        if (!$flightNumber){

            $findFlightNumbers = FlightNumber::where("airline_id", $airline->id)
                                                ->where("flight_number", $messageDetails["flightNumber"])
                                                ->where("departure_airport_id", $foundDepartureAirportId)
                                                ->where("arrival_airport_id", $foundArrivalAirportId)
                                                ->whereNull("deleted_at")
                                                ->get();

            if ($findFlightNumbers->count() > 0){
                $flightNumber = $findFlightNumbers->first();
            }
            // If Doesn't exists or Many found -> create new
            else {
                $flightNumber = new FlightNumber();
                $flightNumber->airline_id = $airline->id;
                $flightNumber->flight_number = ltrim($messageDetails["flightNumber"], "0");
                $flightNumber->departure_airport_id = $foundDepartureAirportId;
                $flightNumber->arrival_airport_id = $foundArrivalAirportId;
                $flightNumber->save();
            }
        }


        /*
        if (!$flight){
            $findFlightNumbers = FlightNumber::where("airline_id", $airline->id)
                                                ->where("flight_number", $messageDetails["flightNumber"])
                                                ->get();

            foreach ($findFlightNumbers as $fnEach) {
                $flight = Flight::where("flight_number_id", "=", $fnEach->id)
//                ->where("aircraft_id", "=", $aircraft->id)
                    ->whereNull("cancelled_at")
                    ->whereNull("deleted_at")
                    ->whereRaw("((s_flights.atd IS NOT NULL AND s_flights.atd != '' AND DATE(s_flights.atd) = '$date') OR
                         (s_flights.etd IS NOT NULL AND s_flights.etd != '' AND DATE(s_flights.etd) = '$date') OR
                         (s_flights.ptd IS NOT NULL AND s_flights.ptd != '' AND DATE(s_flights.ptd) = '$date') OR
                         (s_flights.std IS NOT NULL AND s_flights.std != '' AND DATE(s_flights.std) = '$date'))
                         {$arrivalSearch}")
                    ->first();

                // If Flight Found -> Break
                if ($flight){
                    $flightNumber = $fnEach;
                    break;
                }
            }

            if (!$flightNumber){
                $flightNumber = $findFlightNumbers->first();
            }
        }
        */

        if (!$flight && $arrDiversion){
            $flight = Flight::where("flight_number_id", "=", $flightNumber->id)
//                ->where("aircraft_id", "=", $aircraft->id)
                ->whereNull("cancelled_at")
                ->whereNull("deleted_at")
                ->whereRaw("((s_flights.atd IS NOT NULL AND s_flights.atd != '' AND DATE(s_flights.atd) = '$date') OR
                         (s_flights.etd IS NOT NULL AND s_flights.etd != '' AND DATE(s_flights.etd) = '$date') OR
                         (s_flights.ptd IS NOT NULL AND s_flights.ptd != '' AND DATE(s_flights.ptd) = '$date') OR
                         (s_flights.std IS NOT NULL AND s_flights.std != '' AND DATE(s_flights.std) = '$date'))")
                ->first();
        }

        if ($flight){

            $flight->edited_by_email = $parseEmail && $parseEmail->messageType ? $parseEmail->messageType : 1;

            // Update Aircraft
            self::updateAircraft($aircraft, $flight);
        }
        else{

            // Meaning that flight number might not match, even though the sector arrival or departure
            // airports will match
            $disable = true;
            if ($aircraft && !$disable){
                // Find Flight with This Aircraft Reg
                $flight = Flight::where("aircraft_id", "=", $aircraft->id)
                                    ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                                    ->whereNull("flights.cancelled_at")
                                    ->whereNull("flights.deleted_at");

                // GET SAME SECTOR FLIGHT
                if ($flightNumber) {
                    if ($flightNumber->departure_airport_id) {
                        $flight->where("flights__numbers.departure_airport_id", $flightNumber->departure_airport_id);
                    }

                    if ($flightNumber->arrival_airport_id) {
                        $flight->where("flights__numbers.arrival_airport_id", $flightNumber->arrival_airport_id);
                    }
                }
                // ---------------------

                $flight = $flight->whereRaw("(
                             (s_flights.atd IS NOT NULL AND s_flights.atd != '' AND DATE(s_flights.atd) = '$date') OR
                             (s_flights.etd IS NOT NULL AND s_flights.etd != '' AND DATE(s_flights.etd) = '$date') OR
                             (s_flights.ptd IS NOT NULL AND s_flights.ptd != '' AND DATE(s_flights.ptd) = '$date') OR
                             (s_flights.std IS NOT NULL AND s_flights.std != '' AND DATE(s_flights.std) = '$date'))")
                    ->first(['flights.*']);

                // If Flight Found Ignore This Email
                if ($flight){

                    // Update Aircraft
                    self::updateAircraft($aircraft, $flight);

                    return [
                        "flight"        => $flight,
                        "aircraft"      => $aircraft,
                        "airline"       => $airline,
                        "flightNumber"  => $flightNumber,
                        "codeShare"     => true,
                        "messageLog"    => $messageLog,
                    ];
                }
            }

            $messageLog[] = "Flight Not Found. Flight Was Created by Email.";

            if ($parseEmail){
                $parseEmail->messageLog[] = "Flight Not Found. Flight Was Created by Email.";
            }

            $flight = new Flight();
            // Create flight and FIX timing IF ATA/TDN > CURRENT TIME
            if (self::findAndUpdateFlightsTimings($flight, $messageDetails, $date)) {
                debug("FLIGHT DATE = ".$date);
                $flight->created_by_email = $parseEmail && $parseEmail->messageType ? $parseEmail->messageType : 1;
                $flight->created_by = Auth::user() ? Auth::user()->id : null;
                $flight->flight_number_id = $flightNumber ? $flightNumber->id : null;
                $flight->aircraft_id = $aircraft ? $aircraft->id : null;
                $flight->email_type = $parseEmail ? $parseEmail->messageType : null;
                $flight->email_id = $parseEmail ? $parseEmail->messageId : null;
                $flight->save();

                // Email Notification for Auto-created flights
                try {
                    Parse::sendAutoCreatedEmailNotification($flight);
                } catch (\Exception $e) {
                    Parse::sendSimpleEmailNotification("Delay Email Notification Error" . "\n\n Error: \n" . $e->getMessage());
                }
            }
            else {
                $flight = null;
            }
        }

        // Flight not created
        if (!$flight){
            return [
                "flight"        => null,
                "aircraft"      => $aircraft,
                "airline"       => $airline,
                "flightNumber"  => $flightNumber,
                "codeShare"     => false,
                "messageLog"    => $messageLog,
            ];
        }

        // Process Arrival Diversion
        self::processDiversion($arrDiversion, $flight, $foundArrivalAirportId, $parseEmail);

        return [
            "flight"        => $flight,
            "aircraft"      => $aircraft,
            "airline"       => $airline,
            "flightNumber"  => $flightNumber,
            "codeShare"     => false,
            "messageLog"    => $messageLog,
        ];
    }

    public static function processDiversion($arrDiversion, $divertedFlight, $divertedAirportId, $parseEmail){

        if ($arrDiversion && $divertedAirportId){
            // Flight to original destination

            // Commented this on Jul 6,
            $flightToOriginalDestination = self::createFlightToOriginalDestination($divertedFlight, $divertedAirportId, $parseEmail);

            // Update Current Flight
            $divertedFlight->is_diversion = true;
            $divertedFlight->arrival_airport_id = $divertedAirportId;
            $divertedFlight->diverted_airport_id = $divertedAirportId;
            // Commented this on Jul 6,
            $divertedFlight->next_flight_id = $flightToOriginalDestination ? $flightToOriginalDestination->id : null;
            $divertedFlight->save();

            // Commented this on Jul 6,
            // self::sendDiversionEmailNotification($divertedFlight);
        }
        else {

            if ($divertedFlight->nextFlight){
                $divertedFlight->nextFlight->deleted_at = date("Y-m-d H:i:s");
                $divertedFlight->nextFlight->save();
            }

            $divertedFlight->is_diversion = null;
            $divertedFlight->diverted_airport_id = null;
            $divertedFlight->arrival_airport_id = null;
            $divertedFlight->save();
        }
    }

    public static function createFlightToOriginalDestination($divertedFlight, $divertedAirportId, $parseEmail){

        $flight = Flight::where("prev_flight_id", $divertedFlight->id)
            ->first();

        // Create if not found
        $flightCreated = false;

        if (!$flight) {
            $flight = new Flight();
            $flight->prev_flight_id = $divertedFlight->id;
            $flightCreated = true;
        }

        // Mark as auto-created
        $flight->created_by_email = $parseEmail && $parseEmail->messageType ? $parseEmail->messageType : 1;
        $flight->created_by = Auth::user() ? Auth::user()->id : null;
        $flight->email_type = $parseEmail ? $parseEmail->messageType : null;
        $flight->email_id = $parseEmail ? $parseEmail->messageId : null;

        // Update key fields
        $flight->flight_number_id = $divertedFlight->flight_number_id;

        $flight->departure_airport_id = $divertedAirportId;
        $flight->diverted_airport_id = $divertedAirportId;
        $flight->is_diversion = true;

        $flight->aircraft_id = $divertedFlight->aircraft_id;
        $flight->aircraft_type_id = $divertedFlight->aircraft_type_id;

        $flight->etd = Add_Minutes_To_DateTime(getFlightArrivalDate($divertedFlight), 30);
        $flight->eta = Add_Minutes_To_DateTime(getFlightDepartureDate($flight), 120);

        setFlightDepartureDate($flight);
        setFlightArrivalDate($flight);

        $flight->deleted_at = null;
        $flight->cancelled_at = null;
        $flight->save();

        if ($flightCreated){
            self::sendAutoCreatedEmailNotification($flight);
        }

        return $flight;
    }

    public static function updateAircraft($aircraft, $flight){
        if ($aircraft && $flight && !$aircraft->aircraft_type_id && $flight->aircraft_type_id)
        {
            $aircraft->aircraft_type_id = $flight->aircraft_type_id;
            $aircraft->save();
        }
    }

    public static function updateFlightInformation(&$flight, $messageDetails, &$aircraft, $date, $mvtType = null, $codeShare = false, ParseEmail &$parseEmail = null)
    {
        if (!$flight){

            if ($parseEmail){
                $parseEmail->messageLog[] = "Flight Not Found";
            }

            return;
        }

        if ($codeShare){

            if ($parseEmail){
                $parseEmail->messageLog[] = "Code-share Flight";
            }

            return;
        }

        $now = date("Y-m-d H:i");

        $skipMessage = false;


        // FIRST PROCESS ATD
        if (isset($messageDetails[ATD]) && $messageDetails[ATD])
        {
            $atd = self::makeDateTime($messageDetails[ATD], $date);
            debug("compiled ATD: ". $atd. " NOW: ".$now);
            // 3 mins (180s) margin
            if (strtotime($atd) <= strtotime($now) + 180){
                $flight->atd = $atd;
            }
            else {
                $skipMessage = true;
                debug("MESSAGE SKIPPED");
            }
        }

        // Do not update when ATD > NOW // Skip everything
        if ($skipMessage){
            return;
        }

        if (isset($messageDetails[ETD]) && $messageDetails[ETD])
        {
            $flight->etd = self::makeDateTime($messageDetails[ETD], $date);
        }
        if (isset($messageDetails[ABN]) && $messageDetails[ABN])
        {
            $abn = self::makeDateTime($messageDetails[ABN], $date);
            if (strtotime($abn) <= strtotime($now)){
                $flight->abn = $abn;
            }
        }

        // Get Departure
        $dep = getFlightDepartureDate($flight);

        // FIRST PROCESS ATA
        if (isset($messageDetails[ATA]) && $messageDetails[ATA])
        {
            $ata = self::makeDateTime($messageDetails[ATA], $date);
            if (strtotime($ata) < strtotime($dep)){
                $ata = date("Y-m-d H:i", strtotime(" + 1 days", strtotime($ata)));
            }

            // 3 mins (180s) margin
            if (strtotime($ata) <= (strtotime($now) + 180)){
                $flight->ata = $ata;
            }
            else {
                $skipMessage = true;
                debug("MESSAGE SKIPPED");
            }
            debug("ARR after = ".$ata);
            debug("NOW + 3 mins = ".date("Y-m-d H:i", (strtotime($now) + 180)));
        }

        if ($skipMessage){
            return;
        }

        if (isset($messageDetails[ETA]) && $messageDetails[ETA])
        {
            $eta = self::makeDateTime($messageDetails[ETA], $date);
            if (strtotime($eta) > strtotime($dep)){
                $flight->eta = $eta;
            }
            else {
                $flight->eta = date("Y-m-d H:i", strtotime(" + 1 days", strtotime($eta)));
            }
        }

        if (isset($messageDetails[TDN]) && $messageDetails[TDN])
        {
            $tdn = self::makeDateTime($messageDetails[TDN], $date);
            if (strtotime($tdn) < strtotime($dep)){
                $tdn = date("Y-m-d H:i", strtotime(" + 1 days", strtotime($tdn)));
            }

            if (strtotime($tdn) <= strtotime($now)){
                $flight->tdn = $tdn;
            }
        }

        $initDate = getFlightDepartureInitialDate($flight);
        $actDate = getFlightDepartureDate($flight);

        // in minutes
        $diff = Calculate_Duration($initDate, $actDate, true) * 60;

        if ((isset($messageDetails["delayCodes"]) && $messageDetails["delayCodes"]))
        {
            $delayCodes = explode("/", $messageDetails["delayCodes"]);
            $timeStr = explode("/", $messageDetails["delayTimes"]);

            $delayTimes = [];

            foreach ($delayCodes as $j => $each ) {
                if (isset($timeStr[$j]) && $timeStr[$j]){
                    $dTime = self::makeTime(trim($timeStr[$j]));

                    debug("REAL: ".$timeStr[$j]." CONVERTED:".$dTime);
                }
                else {

                    // If 1 delay mentioned assign all delay time to this delay
                    if (count($delayCodes) == 1){
                        $dTime = Minutes_To_Time($diff);
                    }
                    else {

                        if ($j == 0){
                            $dTime = Minutes_To_Time($diff - (count($delayCodes) - 1));
                        }
                        else {
                            $dTime = self::makeTime("0001");
                        }
                    }
                }

                $delayTimes[] = $dTime;
            }

            if (isset($messageDetails["extDelayCode"]) && $messageDetails["extDelayCode"] &&
                isset($messageDetails["extDelayTime"]) && $messageDetails["extDelayTime"]){

                $delayCodes[] = trim($messageDetails["extDelayCode"]);
                $delayTimes[] = self::makeTime(trim($messageDetails["extDelayTime"]));;

            }

            Flight::SetFlightDelays($flight, $delayCodes, $delayTimes);
        }
        else if (isset($messageDetails["delayDescriptions"]) && $messageDetails["delayDescriptions"]){
            if ($timeStr = $messageDetails["delayTimes"]){

                if (strlen($timeStr) != 4) {

                    if (contains($timeStr, "ч") || contains($timeStr, "мин"))
                    {
                        if (!contains($timeStr, "ч")) {
                            $timeStr = "00" . $timeStr;
                        }
                        if (!contains($timeStr, "мин")) {
                            $timeStr .= "00";
                        }
                    }

                    $timeStr = str_replace(["ч", "мин"], "", $timeStr);
                }

                $dTime = self::makeTime(trim($timeStr));
            }
            else {
                $dTime = Minutes_To_Time($diff);
            }

            Flight::SetFlightDelaysNoCode($flight, $dTime, $messageDetails["delayDescriptions"]);
        }
        else {
            if ($mvtType && $mvtType == MVT_DEPARTURE){
                // Remove All Delays
                $fDelay = FlightDelay::where("flight_id", $flight->id)
                                    ->get();

                // Audit Deleted
                auditDeleted($fDelay);
            }
        }

        if (isset($messageDetails["baggage"]))
        {
            $flight->baggage = getIntValueOrNull($messageDetails["baggage"]);
        }
        if (isset($messageDetails["cargo"]))
        {
            $flight->cargo = getIntValueOrNull($messageDetails["cargo"]);
        }
        if (isset($messageDetails["mail"]))
        {
            $flight->mail = getIntValueOrNull($messageDetails["mail"]);
        }
        if (isset($messageDetails["eic"]))
        {
            $flight->eic = getIntValueOrNull($messageDetails["eic"]);
        }
        if ((array_key_exists("capacityC", $messageDetails) || array_key_exists("capacityY", $messageDetails))
            && (isset($messageDetails["capacityC"]) || isset($messageDetails["capacityY"]))) {

            if (array_key_exists("capacityC", $messageDetails)) {
                $flight->capacity_c = getIntValueOrNull($messageDetails["capacityC"]);
            }
            if (array_key_exists("capacityY", $messageDetails)) {
                $flight->capacity_y = getIntValueOrNull($messageDetails["capacityY"]);
            }
        }
        if (isset($messageDetails["paxC_bkd"]))
        {
            $flight->pax_c_booked = getIntValueOrNull($messageDetails["paxC_bkd"]);
        }
        if (isset($messageDetails["paxY_bkd"]))
        {
            $flight->pax_y_booked = getIntValueOrNull($messageDetails["paxY_bkd"]);
        }
        if (isset($messageDetails["paxC"]))
        {
            $flight->pax_c_actual = getIntValueOrNull($messageDetails["paxC"]);
        }
        if (isset($messageDetails["paxY"]))
        {
            $flight->pax_y_actual = getIntValueOrNull($messageDetails["paxY"]);
        }
        if (array_key_exists("paxInf", $messageDetails))
        {
            $flight->pax_inf_actual = getIntValueOrNull($messageDetails["paxInf"]);
        }
        if (array_key_exists("paxAdults", $messageDetails))
        {
            $flight->pax_adults_actual = getIntValueOrNull($messageDetails["paxAdults"]);
        }
        if (array_key_exists("paxM", $messageDetails))
        {
            $flight->pax_m_actual = getIntValueOrNull($messageDetails["paxM"]);
        }
        if (array_key_exists("paxF", $messageDetails))
        {
            $flight->pax_f_actual = getIntValueOrNull($messageDetails["paxF"]);
        }
        if (array_key_exists("paxCh", $messageDetails))
        {
            $flight->pax_ch_actual = getIntValueOrNull($messageDetails["paxCh"]);
        }
        if (isset($messageDetails["paxMvtTotal"]))
        {
            $flight->pax_mvt_total = getIntValueOrNull($messageDetails["paxMvtTotal"]);
        }
        if (isset($messageDetails["paxMvtInf"]))
        {
            $flight->pax_mvt_inf = getIntValueOrNull($messageDetails["paxMvtInf"]);
        }

        if (isset($messageDetails["totalWt"]))
        {
            $flight->cmp_total_wt = getIntValueOrNull($messageDetails["totalWt"]);
        }
        if (isset($messageDetails["comp1Wt"]))
        {
            $flight->cmp_1_wt = getIntValueOrNull($messageDetails["comp1Wt"]);
        }
        if (isset($messageDetails["comp2Wt"]))
        {
            $flight->cmp_2_wt = getIntValueOrNull($messageDetails["comp2Wt"]);
        }
        if (isset($messageDetails["comp3Wt"]))
        {
            $flight->cmp_3_wt = getIntValueOrNull($messageDetails["comp3Wt"]);
        }
        if (isset($messageDetails["comp4Wt"]))
        {
            $flight->cmp_4_wt = getIntValueOrNull($messageDetails["comp4Wt"]);
        }
        if (isset($messageDetails["comp5Wt"]))
        {
            $flight->cmp_5_wt = getIntValueOrNull($messageDetails["comp5Wt"]);
        }
        // New Fields
        if (isset($messageDetails["comp5Wt"]))
        {
            $flight->cmp_5_wt = getIntValueOrNull($messageDetails["comp5Wt"]);
        }
        if (array_key_exists("fob", $messageDetails))
        {
            $flight->fob = $messageDetails["fob"];
        }
        if (array_key_exists("in_tanks", $messageDetails))
        {
            $flight->in_tanks = getIntValueOrNull($messageDetails["in_tanks"]);
        }
        if (array_key_exists("tof", $messageDetails))
        {
            $flight->tof = $messageDetails["tof"];
        }
        if (array_key_exists("eet", $messageDetails))
        {
            $flight->eet = makeTime($messageDetails["eet"]);
        }
        if (array_key_exists("rf", $messageDetails))
        {
            $flight->rf = $messageDetails["rf"];

            $departureFuel = $flight->fob ? $flight->fob : ($flight->in_tanks + ($flight->uplifted ? $flight->uplifted : 0));

            // correct utilised figures
            if ($flight->rf && $departureFuel){
                $flight->utilised = $departureFuel - $flight->rf;
            }
        }
        if (array_key_exists("pic", $messageDetails))
        {
            $flight->pic = $messageDetails["pic"];
        }

        if ($aircraft)
        {
            $flight->aircraft_id = $aircraft->id;

            if (!$aircraft->config_c && !$aircraft->config_y){
                if (isset($messageDetails["capacityC"]) && $messageDetails["capacityC"]){
                    $aircraft->config_c = getIntValueOrNull($messageDetails["capacityC"]);
                }
                if (isset($messageDetails["capacityY"]) && $messageDetails["capacityY"]){
                    $aircraft->config_y = getIntValueOrNull($messageDetails["capacityY"]);
                }
                $aircraft->save();
            }
        }

        // Correct Flight Timings
        try{
            self::checkAndCorrectFlightTimings($flight);
        }
        catch(\Exception $e){}

        $flight->save();
    }

    public static function checkAndCorrectFlightTimings(&$flight){

        $departureTimes = [
            ABN,
            ATD,
            ETD,
//            PTD,
        ];

        $arrivalTimes = [
//            PTA,
            ETA,
            TDN,
            ATA
        ];

        $depPoint = $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : $flight->ptd;

        foreach ( $departureTimes as $dep) {
            if ($flight->{$dep} && $flight->{$dep} != EMPTY_DATETIME){
                $depPoint = $flight->{$dep};
                break;
            }
        }

        debug("Check and Correct Flight Timings. DepPoint {$depPoint}");

        if (!$depPoint){
            return;
        }

        $depPointDay = date("Y-m-d", strtotime($depPoint));
        $nextDay = date("Y-m-d", strtotime(" + 1 days", strtotime($depPointDay)));

        $now = date("Y-m-d H:i:s");

        foreach ($arrivalTimes as $i => $arr) {
            if ($flight->{$arr} && $flight->{$arr} != EMPTY_DATETIME && (strtotime($flight->{$arr}) < strtotime($depPoint))){
                debug("DEP {$depPoint}. ARR Before {$flight->$arr}");
                $arrDateTime = $nextDay." ".date("H:i:s", strtotime($flight->{$arr}));
                if ( strtotime($arrDateTime) <= strtotime($now)){
                    $flight->{$arr} = $arrDateTime;
                }
            }
        }

        if (($flight->etd && $flight->etd != EMPTY_DATETIME) && (!$flight->eta || $flight->eta == EMPTY_DATETIME)){

            if ($flight->ptd && $flight->ptd != EMPTY_DATETIME && $flight->pta && $flight->pta != EMPTY_DATETIME){
                $flightDuration = Calculate_Duration($flight->ptd, $flight->pta);

                $arrDateTime =  Add_Minutes_To_DateTime($flight->etd, $flightDuration, true);
                //if ( strtotime($arrDateTime) <= strtotime($now)){
                $flight->eta = $arrDateTime;
                //}
            }
            else  if ($flight->std && $flight->std != EMPTY_DATETIME && $flight->sta && $flight->sta != EMPTY_DATETIME){
                $flightDuration = Calculate_Duration($flight->std, $flight->sta);

                $arrDateTime =  Add_Minutes_To_DateTime($flight->etd, $flightDuration, true);
                //if ( strtotime($arrDateTime) <= strtotime($now)){
                $flight->eta = $arrDateTime;
                //}
            }

//            debug("Flight ETA: ". $flight->eta. " based on ETD: ".$flight->etd);
        }


        $dep = getFlightDepartureDate($flight); //0920
        $arr = getFlightArrivalDate($flight); // 1320
        // IF Arrival timings are less than departure
        // Then auto-fill ETA for correct flight box in Flight Watch
        if (strtotime($arr) < strtotime($dep)){

//            debug("ARR < DEP ERROR. ARR: ".$arr. " | DEP: ".$dep );

            if ($flight->ptd && $flight->ptd != EMPTY_DATETIME && $flight->pta && $flight->pta != EMPTY_DATETIME){
                $flightDuration = Calculate_Duration($flight->ptd, $flight->pta);

                $estDateTime =  Add_Minutes_To_DateTime($dep, $flightDuration, true);
                //if ( strtotime($estDateTime) <= strtotime($now)){
                    $flight->eta = $estDateTime;
                //}
                debug("PTD: {$flight->ptd}");
            }
            else  if ($flight->std && $flight->std != EMPTY_DATETIME && $flight->sta && $flight->sta != EMPTY_DATETIME){
                $flightDuration = Calculate_Duration($flight->std, $flight->sta);

                $estDateTime =  Add_Minutes_To_DateTime($dep, $flightDuration, true);
                //if ( strtotime($estDateTime) <= strtotime($now)){
                    $flight->eta = $estDateTime;
                //}
                debug("STD: {$flight->std}");
            }
        }

        setFlightDepartureDate($flight);
        setFlightArrivalDate($flight);

    }

    public static function createAirport($iata){
        if (!$iata){
            return null;
        }

        $record = Airport::checkAndGet($iata);

        if ($record){
            return $record;
        }

        $airport = Airport::findOrCreateByIATA($iata);


        return $airport;
    }

    private static function updateFlightNumber(&$flightNumber, $messageDetails, $airportIds){
        $updated = false;

        if (!$airportIds){
            $airportIds = Airport::whereNotNull("iata")->pluck("id", "iata")->all();
        }

        //  Departure Airport
        if (!$flightNumber->departure_airport_id) {
            if (isset($messageDetails["departureAirport"]) && $messageDetails["departureAirport"]) {
                if (isset($airportIds[$messageDetails["departureAirport"]])) {
                    $flightNumber->departure_airport_id = $airportIds[$messageDetails["departureAirport"]];
                } else {
                    $depAirport = self::createAirport($messageDetails["departureAirport"]);
                    $flightNumber->departure_airport_id = $depAirport ? $depAirport->id : null;
                }
                $updated = true;
            }
        }

        // Arrival Airport
        if (!$flightNumber->arrival_airport_id) {
            if (isset($messageDetails["arrivalAirport"]) && $messageDetails["arrivalAirport"]) {
                if (isset($airportIds[$messageDetails["arrivalAirport"]])) {
                    $flightNumber->arrival_airport_id = $airportIds[$messageDetails["arrivalAirport"]];
                } else {
                    $arrAirport = self::createAirport($messageDetails["arrivalAirport"]);
                    $flightNumber->arrival_airport_id = $arrAirport ? $arrAirport->id : null;
                }
            }
        }

        if ($updated){
            $flightNumber->save();
        }
    }

    public static function findAndUpdateFlightsTimings(&$flight, $messageDetails, $date){

        $departure = self::departureTimeExists($messageDetails, $date);
        $arrival = self::arrivalTimeExists($messageDetails, $date);

        if (! $departure && !$arrival){
            $flight->ptd = $date." 01:00:00";
            $flight->pta = $date." 04:00:00";
        }
        else {
            $now = date("Y-m-d H:i");

            if ($departure) {

                $depType = $departure[0];
                $depTime = $departure[1];
                $arrType = $departure[2];

                // 3 mins margin
                if (!in_array($depType, [ATD, ABN]) ||
                    (in_array($depType, [ATD, ABN]) && (strtotime($depTime) <= strtotime($now) + 180)))
                {
                    $flight->{$depType} = $depTime;
                }

                // If No Arrival Time Found -> add 1hr to departure time
                if (!$arrival) {

                    $arrTime = self::addMinutesToDateTime($depTime, 120);
                    // $departure[2] -> related arrival timing

                    // 3 mins margin
                    if (!in_array($arrType, [TDN, ATA])
                    || (in_array($arrType, [TDN, ATA]) &&  (strtotime($arrTime) <= strtotime($now) + 180)))
                    {
                        $flight->{$arrType} = $arrTime;
                    }

                }
            }

            if ($arrival) {

                $arrType = $arrival[0];
                $arrTime = $arrival[1];
                $depType = $arrival[2];

                // 3 mins margin
                if (in_array($arrType, [TDN, ATA]) && (strtotime($arrTime) >= strtotime($now) + 180)){
                    debug("FLIGHT NOT CREATED. ATA/TDN - ". $arrTime);
                    return false;
                }
                else {
                    // Arrival

                    if (isset($depTime) && $depTime && strtotime($depTime) > strtotime($arrTime)){
                        debug("ARR Time {$arrTime} < DEP {$depTime}". " ADDED + 1 days to ARR");
                        $arrTime = date("Y-m-d H:i", strtotime("+ 1 days", strtotime($arrTime)));
                    }

                    $flight->{$arrType} = $arrTime;

                    debug("ARR TYPE:" . $arrType);
                    // If No Departure Time Found -> subtract 1hr from arrival time

                    if (!$departure) {
                        $depTime = self::subtractMinutesToDateTime($arrTime, 120);

                        // 3 mins margin
                        if (!in_array($depType, [ATD, ABN]) ||
                            (in_array($depType, [ATD, ABN]) && (strtotime($depTime) <= strtotime($now) + 180))
                        ) {
                            $flight->{$depType} = $depTime;
                        }
                    }
                }
            }

            /*
             * Check if STD/PTD Date is not there and DAY < ATD/ETD
             * EX
             * MVT
                7J630/09.EY756.DME
                AD100250/100322 EA100703 DYU
             *
             */
            $dep = getFlightDepartureInitialDate($flight, true);
            if (!$flight->std && !$flight->ptd && strtotime($date) < strtotime($dep)){
                $flight->ptd = $date." 01:00:00";
                $flight->pta = $date." 04:00:00";
            }
            // Edited on Aug 6 2019
            else {
                if ($flight->atd || $flight->abn || $flight->etd){
                    $flight->ptd = $flight->atd ? $flight->atd : ($flight->abn ? $flight->abn : $flight->etd);
                }
            }
        }

        setFlightDepartureDate($flight);
        setFlightArrivalDate($flight);

        return true;
    }

    public static function addMinutesToDateTime($dateTime, $minutes){
        return date("Y-m-d H:i", strtotime("+ {$minutes} minutes", strtotime($dateTime)));
    }

    public static function subtractMinutesToDateTime($dateTime, $minutes){
        return date("Y-m-d H:i", strtotime("- {$minutes} minutes", strtotime($dateTime)));
    }

    public static function findAircraft($aircraftName, Airline $airline = null, $aircraftType = null){
        $aircraft = null;

        if ($aircraftName != "" && $airline){
            $aircraft = Aircraft::where("airline_id", $airline->id);

            if (contains($aircraftName, "-")){
                $aircraft->where("name",$aircraftName);
            }
            else {
                $aircraft->where("mvt_name", $aircraftName);
            }

            $aircraft = $aircraft->first();

            if (!$aircraft){

                if (!partitionAircraftCode($aircraftName)){
                    return null;
                }

                $aircraft = new Aircraft();
                $aircraft->mvt_name         = contains($aircraftName, "-") ? str_replace("-", "", $aircraftName) : $aircraftName;
                $aircraft->name             = contains($aircraftName, "-") ? $aircraftName : partitionAircraftCode($aircraftName);
                $aircraft->airline_id       = $airline->id;
            }

            if ($aircraftType){
                $acType = AircraftType::findByName($aircraftType, $airline);

                if ($acType){
                    $aircraft->aircraft_type_id = $acType->id;
                }
            }

            $aircraft->deleted_at = null;
            $aircraft->save();
        }

        return $aircraft;
    }

    public static function arrivalTimeExists($messageDetails, $date){
        if( (isset($messageDetails[STA]) && $messageDetails[STA]) ||
            (isset($messageDetails[PTA]) && $messageDetails[PTA]) ||
            (isset($messageDetails[ETA]) && $messageDetails[ETA]) ||
            (isset($messageDetails[ATA]) && $messageDetails[ATA]) ||
            (isset($messageDetails[TDN]) && $messageDetails[TDN]) ){

            if (isset($messageDetails[STA]) && $messageDetails[STA]){
                return [STA, self::makeDateTime($messageDetails[STA], $date), STD];
            }
            else if (isset($messageDetails[PTA]) && $messageDetails[PTA]){
                return [PTA, self::makeDateTime($messageDetails[PTA], $date), PTD];
            }
            elseif (isset($messageDetails[ETA]) && $messageDetails[ETA]){
                return [ETA, self::makeDateTime($messageDetails[ETA], $date), ETD];
            }
            elseif (isset($messageDetails[ATA]) && $messageDetails[ATA]){
                return [ATA, self::makeDateTime($messageDetails[ATA], $date), ETD];
            }
            elseif (isset($messageDetails[TDN]) && $messageDetails[TDN]){
                return [TDN, self::makeDateTime($messageDetails[TDN], $date), ABN];
            }
        }

        return null;
    }

    public static function departureTimeExists($messageDetails, $date){
        if((isset($messageDetails[STD]) && $messageDetails[STD]) ||
            (isset($messageDetails[PTD]) && $messageDetails[PTD]) ||
            (isset($messageDetails[ETD]) && $messageDetails[ETD]) ||
            (isset($messageDetails[ATD]) && $messageDetails[ATD]) ||
            (isset($messageDetails[ABN]) && $messageDetails[ABN]) ){

            if (isset($messageDetails[STD]) && $messageDetails[STD]){
                return [STD, self::makeDateTime($messageDetails[STD], $date), STA];
            }
            else if (isset($messageDetails[PTD]) && $messageDetails[PTD]){
                return [PTD, self::makeDateTime($messageDetails[PTD], $date), PTA];
            }
            elseif (isset($messageDetails[ETD]) && $messageDetails[ETD]){
                return [ETD, self::makeDateTime($messageDetails[ETD], $date), ETA];
            }
            elseif (isset($messageDetails[ATD]) && $messageDetails[ATD]){
                return [ATD, self::makeDateTime($messageDetails[ATD], $date), ETA];
            }
            else {
                return [ABN, self::makeDateTime($messageDetails[ABN], $date), TDN];
            }
        }

        return null;
    }

    public static function makeDateTime($str, $date = null){

        if (strlen($str) == 6){
            list($day, $hours, $minutes) = str_split($str, 2);
            if ($date){
                return date("Y-m", strtotime($date)) . "-{$day} {$hours}:{$minutes}:00";
            }
            else {
                return date("Y-m" . "-{$day} {$hours}:{$minutes}:00");
            }
        }
        else if (strlen($str) == 4)
        {
            list($hours, $minutes) = str_split($str, 2);
            if ($date){
                return date("Y-m-d H:i", strtotime("{$date} {$hours}:{$minutes}:00"));
            }
            else {
                return date("Y-m-d ". "{$hours}:{$minutes}:00");
            }
        }
//        $this->error("Cannot get date string from string - $str and departure day - $date");
        return "";
        //return date("Y-m-d H:i", strtotime($date." ".$HHmm));
    }

    public static function makeTime($str){

        if (strlen($str) == 4)
        {
            list($hours, $minutes) = str_split($str, 2);

            return date("H:i:s", strtotime("{$hours}:{$minutes}:00"));
        }

        preg_match('/(\d{2}:\d{2}(?:\d{2})?)/', $str, $match);

        return isset($match[1]) ? $str : null;
    }

    public static function getBaggageMailCargo($bcm, &$messageDetails){
        if (count($bcm)){
            foreach ($bcm as $type => $value) {

                switch ($type){
                    case "B":
                    case "BAG":
                        $wtPcs = explode("/", $value);
                        if ($wtPcs && count($wtPcs)){
                            if (count($wtPcs) == 1){
                                $messageDetails["baggage"] = (isset($messageDetails["baggage"]) && trim($messageDetails["baggage"]) ? trim($messageDetails["baggage"]) : 0) +  trim($wtPcs[0]);
                            }
                            else {
                                if (isset($wtPcs[0])){
                                    $messageDetails["baggage_pcs"] = (isset($messageDetails["baggage_pcs"]) && trim($messageDetails["baggage_pcs"]) ? trim($messageDetails["baggage_pcs"]) : 0) + trim($wtPcs[0]);
                                }
                                if (isset($wtPcs[1])){
                                    $messageDetails["baggage"] = (isset($messageDetails["baggage"]) && trim($messageDetails["baggage"]) ? trim($messageDetails["baggage"]) : 0) +  trim($wtPcs[1]);
                                }
                            }
                        }
                        break;
                    case "C":
                    case "FRE":
                        $wtPcs = explode("/", $value);
                        if ($wtPcs && count($wtPcs)){
                            if (count($wtPcs) == 1){
                                $messageDetails["cargo"] = (isset($messageDetails["cargo"]) && trim($messageDetails["cargo"]) ? trim($messageDetails["cargo"]) : 0) +  trim($wtPcs[0]);
                            }
                            else {
                                if (isset($wtPcs[0])){
                                    $messageDetails["cargo_pcs"] = (isset($messageDetails["cargo_pcs"]) && trim($messageDetails["cargo_pcs"]) ? trim($messageDetails["cargo_pcs"]) : 0) + trim($wtPcs[0]);
                                }
                                if (isset($wtPcs[1])){
                                    $messageDetails["cargo"] = (isset($messageDetails["cargo"]) && trim($messageDetails["cargo"]) ? trim($messageDetails["cargo"]) : 0) +  trim($wtPcs[1]);
                                }
                            }
                        }

//                        $messageDetails["cargo"] = (isset($messageDetails["cargo"]) && trim($messageDetails["cargo"]) ? trim($messageDetails["cargo"]) : 0) + $value;
                        break;

                    case "E":
                        $messageDetails["eic"] = (isset($messageDetails["eic"]) && trim($messageDetails["eic"]) ? trim($messageDetails["eic"]) : 0) + $value;

                        break;
                    case "M":
                    case "POS":
                        $messageDetails["mail"] = (isset($messageDetails["mail"]) && trim($messageDetails["mail"]) ? trim($messageDetails["mail"]) : 0) + $value;
                        break;
                }

            }
        }
    }

    public static function findPassengerProcess($passengerDetails, $flight, $sentDate, $countryIDs = null, $countryISOIDs = null, $act = FALSE, $source = PRL, $strictMatching = false){

        if (!$countryIDs){
            $countryIDs = Country::whereNotNull("abbr")->pluck("id", "abbr")->all();
        }

        if (!$countryISOIDs){
            $countryISOIDs = Country::whereNotNull("iso_alpha_3")->pluck("id", "iso_alpha_3")->all();
        }

        // Find or Create Passenger
        $passenger = self::findOrCreatePassenger($passengerDetails, $flight, $countryIDs, $countryISOIDs, $source, $strictMatching);

        // Find or Create Passenger Order by ticket number
        $passengerOrder = self::findOrCreatePassengerOrder($passengerDetails);

        // Find or Create Passenger Flight
        $passengerFlight = self::findOrCreatePassengerFlight($passenger, $passengerOrder, $flight, $passengerDetails, $sentDate, null, null, $act, $source);

        // Create INF Record
        if (isset($passengerDetails['infLname'])){
            // Find or Create Passenger
            $passengerInf = self::findOrCreateInfPassenger($passengerDetails);

            // Find or Create Passenger Order by ticket number
            $passengerInfOrder = self::findOrCreatePassengerOrder($passengerDetails, TRUE);

            // Find or Create Passenger Flight
            self::findOrCreatePassengerFlight($passengerInf, $passengerInfOrder, $flight, $passengerDetails, $sentDate, TRUE, $passengerFlight, $act, $source);
        }

        return $passengerFlight;
    }

    public static function findInfPassengerProcess($passengerDetails, $flight, $sentDate, $parentPassengerFlight, $act = FALSE, $source = PRL){

        // Create INF Record
        if (isset($passengerDetails['infLname'])){
            // Find or Create Passenger
            $passengerInf = self::findOrCreateInfPassenger($passengerDetails);

            // Find or Create Passenger Order by ticket number
            $passengerInfOrder = self::findOrCreatePassengerOrder($passengerDetails, TRUE);

            // Find or Create Passenger Flight
            return self::findOrCreatePassengerFlight($passengerInf, $passengerInfOrder, $flight, $passengerDetails, $sentDate, TRUE, $parentPassengerFlight, $act, $source);
        }

        return null;
    }


    public static function findOrCreatePassengerFrequentFlyer($passenger, $passengerDetails){

        if (!$passenger || !isset($passengerDetails['ffAirline']) || !$passengerDetails['ffAirline'] || strlen($passengerDetails['ffAirline']) != 2){
            return null;
        }

        // find airline
        $airline = Airline::where("iata", $passengerDetails['ffAirline'])->first();

        if (!$airline){
            $airline = new Airline();
            $airline->iata = strtoupper($passengerDetails['ffAirline']);
            $airline->save();
        }

        $ff = PassengerFrequentFlyer::where("passenger_id", $passenger->id)
            ->where('airline_id', $airline->id)
            ->where('ff_no', $passengerDetails['ffNo'])
            ->first();

        if (!$ff){
            $ff = new PassengerFrequentFlyer();
            $ff->passenger_id = $passenger->id;
            $ff->airline_id = $airline->id;
            $ff->ff_no = $passengerDetails['ffNo'];
            $ff->save();
        }

        return $ff;
    }

    public static function findOrCreatePassenger($passengerDetails, $flight, $countryIDs, $countryISOIDs, $source, $strictMatching = false){
        $passenger = null;

        if (!isset($passengerDetails['lastName'])) {
            return null;
        }

        $createNewRecord = false;

        // Pre Source
        if (in_array($source, [PNL, ADL])){

            if ($flight) {
                $passengerFlight = PassengerFlight::with(["passenger"])
                                                    ->join("passengers", "passengers.id", "=", "passengers__flights.passenger_id")
                                                    ->where("flight_id", $flight->id)
                                                    ->where("passengers.last_name", $passengerDetails['lastName']);

                if (isset($passengerDetails['firstName'])) {
                    $passengerFlight->where("first_name", $passengerDetails['firstName']);
                }

                if (isset($passengerDetails["dob"]) && $passengerDetails["dob"]){
                    $passengerFlight->where(function($sql) use ($passengerDetails){
                                        $sql->where("dob", self::getFormattedDate($passengerDetails['dob'], true))
                                            ->orWhereNull("dob");
                                    });
                }

                $passengerFlight = $passengerFlight->get();

                $seatNum = isset($passengerDetails['seat']) && $passengerDetails['seat'] ? $passengerDetails['seat'] : null;
                if (count($passengerFlight) > 1)
                {
                    $seatExistsInPrevRecords = false;

                    if ($seatNum) {
                        foreach ($passengerFlight as $each) {
                            if ($each->seat_number){
                                $seatExistsInPrevRecords = true;
                            }
                            // Found break
                            if ($each->seat_number == $seatNum) {
                                $passenger = $each->passenger;
                            }
                        }

                        if (!$passenger && $seatExistsInPrevRecords)
                        {
                            $createNewRecord = true;
                        }
                    }

                    // If not found matching and no order to create new record => pick one
                    if (!$passenger && !$createNewRecord)
                    {
                        $passenger = $passengerFlight->first()->passenger;
                    }
                }
                // 1.2 1 Matching
                else if (count($passengerFlight) == 1)
                {
                    $passengerFlight = $passengerFlight->first();

                    if (!$seatNum || !$passengerFlight->seat_number || ($seatNum && $passengerFlight->seat_number == $seatNum))
                    {
                        $passenger = $passengerFlight->passenger;
                    }
                    else {
                        $createNewRecord = true;
                    }
                }
            }

            if (!$passenger && !$createNewRecord) {

                $passenger = Passenger::where("last_name", $passengerDetails["lastName"]);
                if (isset($passengerDetails['firstName'])) {
                    $passenger->where("first_name", $passengerDetails['firstName']);
                }

                if (isset($passengerDetails["dob"]) && $passengerDetails["dob"]){
                    $passenger->where(function($sql) use ($passengerDetails){
                                $sql->where("dob", self::getFormattedDate($passengerDetails['dob'], true))
                                    ->orWhereNull("dob");
                            });
                }

                $passenger = $passenger->first();

            }

        }
        // ACT Source
        else {
            if ($flight) {
                $passengerFlight = PassengerFlightAct::with(["passenger"])
                                                    ->join("passengers", "passengers.id", "=", "passengers__flights_act.passenger_id");

                $lName = $passengerDetails['lastName'];
                if (strlen($lName) > 15) {
                    $lName = substr($passengerDetails['lastName'], 0, 15);
                }

                $passengerFlight->where("flight_id", $flight->id)
                                ->where("passengers.last_name", "LIKE", $lName . "%");

                if (isset($passengerDetails['is_chd'])){
                    $passengerFlight->where("is_chd", $passengerDetails['is_chd']);
                }

                $searchPassengerFlight = clone $passengerFlight;

                if (isset($passengerDetails['firstName'])) {
                    $searchPassengerFlight->where("first_name", "LIKE", $passengerDetails['firstName']);
                }


                // 1.Exact match
                $passengerFlightQuery = $searchPassengerFlight->get();
                $seatNum = isset($passengerDetails['seat']) && $passengerDetails['seat'] ? $passengerDetails['seat'] : null;
                if (count($passengerFlightQuery)) {

                    // 1.1 More than 1 matching
                    if (count($passengerFlightQuery) > 1) {
                        // if seat number exists loop and find matching one
                        $seatExistsInPrevRecords = false;
                        if ($seatNum) {
                            foreach ($passengerFlightQuery as $each) {
                                if ($each->seat_number){
                                    $seatExistsInPrevRecords = true;
                                }
                                // Found break
                                if ($each->seat_number == $seatNum) {
                                    $passenger = $each->passenger;
                                }
                            }

                            if (!$passenger && $seatExistsInPrevRecords)
                            {
                                $createNewRecord = true;
                            }
                        }

                        // If not found matching and no order to create new record => pick one
                        if (!$passenger && !$createNewRecord)
                        {
                            $passenger = $passengerFlightQuery->first()->passenger;
                        }
                    }
                    // 1.2 1 Matching
                    else {
                        $passengerFlightQuery = $passengerFlightQuery->first();

                        if (!$seatNum || !$passengerFlightQuery->seat_number || ($seatNum && $passengerFlightQuery->seat_number == $seatNum))
                        {
                            $passenger = $passengerFlightQuery->passenger;
                        }
                        else {
                            $createNewRecord = true;
                        }
                    }
                }

                // No passenger and no create new record order => go and loook
                if (!$passenger && !$createNewRecord) {

                    $searchPassengerFlight = clone $passengerFlight;

                    if (isset($passengerDetails['firstName'])) {
                        $searchPassengerFlight->where("first_name", "LIKE", str_replace(["  "," "], "", $passengerDetails['firstName']));
                    }

                    // 2.Remove spacing in between of first name
                    if ($passengerFlightQuery = $searchPassengerFlight->first()) {
                        // Found
                        $passenger = $passengerFlightQuery->passenger;
                    }
                    else {
                        // 3. First name substring match only in PTM
                        if (in_array($source, [PTM])) {

                            // Reset var
                            $searchPassengerFlight = clone $passengerFlight;

                            if (isset($passengerDetails['firstName'])) {
                                $length = strlen($passengerDetails['firstName']);
                                $cutLength = $length >= 4 ? $length - 2 : ($length >= 2 ? $length - 1 : $length);
                                $fName = substr($passengerDetails['firstName'], 0, $cutLength);

                                $searchPassengerFlight->where("first_name", "LIKE", $fName . "%");
                            }

                            $passengerFlightQuery = $searchPassengerFlight->first();

                            if ($passengerFlightQuery) {
                                // Found
                                $passenger = $passengerFlightQuery->passenger;
                            }
                            else {
                                if (!$strictMatching && isset($passengerDetails['firstName']) && strlen($passengerDetails['firstName']) > 5) {
                                    $searchPassengerFlight = clone $passengerFlight;

                                    // Find with 4 First Name length
                                    $fName = substr($passengerDetails['firstName'], 0, 5);

                                    $searchPassengerFlight->where("first_name", "LIKE", $fName . "%");

                                    $passengerFlightQuery = $searchPassengerFlight->first();

                                    if ($passengerFlightQuery) {
                                        $passenger = $passengerFlightQuery->passenger;
                                    }
                                }
                            }

                        }
                    }
                }
            }

            // No passenger and no create new record order => go and loook
            if (!$passenger && !$createNewRecord){
                switch($source){
                    case PTM:
                        $passenger = Passenger::where("last_name", "LIKE", $passengerDetails["lastName"]. "%");
                        if (isset($passengerDetails['firstName'])) {
                            $passenger->where("first_name", "LIKE", $passengerDetails['firstName'] . "%");
                        }
                        break;

                    default:
                        $passenger = Passenger::where("last_name", $passengerDetails["lastName"]);
                        if (isset($passengerDetails['firstName'])) {
                            $passenger->where("first_name", $passengerDetails['firstName']);
                        }
                        break;
                }

                $passenger = $passenger->first();

                // Extra search to find matching passenger
                if (!$passenger){

                    $lName = $passengerDetails['lastName'];

                    if (strlen($lName) > 15){
                        $lName =  substr($passengerDetails['lastName'], 0, 15);
                    }

                    // ONLY PTM uses substring matching
                    if (in_array($source, [PTM]))
                    {
                        $passenger = Passenger::where("last_name", "LIKE", $lName . "%");

                        if (isset($passengerDetails['firstName'])) {
                            // include first 2-letters of the first name if first name > 3
                            // otherwise take only initials
                            $length = strlen($passengerDetails['firstName']);
                            $cutLength = $length >= 4 ? $length - 2 : ($length >= 2 ? $length - 1 : $length);
                            $fName = substr($passengerDetails['firstName'], 0, $cutLength);

                            $passenger->where("first_name", "LIKE", $fName . "%");
                        }

                        $passenger = $passenger->first();
                    }
                }

                if ($passenger && $passenger->created_source == PTM && $source != PTM && !$passenger->updated_source){

                    // Update PAX INFO
                    switch($source){
                        case TPM:
                        case PSM:

                            $passenger->updated_source = $source;
                            $passenger->last_name = $passengerDetails["lastName"];

                            if (isset($passengerDetails['firstName'])) {
                                $passenger->first_name = $passengerDetails['firstName'];
                            }

                            $passenger = $passenger->save();

                            break;
                    }
                }
            }
        }


        if (!($passenger && $passenger instanceof Passenger)) {

            if ($passengerDetails["lastName"] == "C" && !$passengerDetails["firstName"]){
                return null;
            }

            $passenger = new Passenger();
            // Set Source
            $passenger->created_source = $source;
            $passenger->last_name = $passengerDetails['lastName'];
            $createdName = $passengerDetails['lastName'];

            if (isset($passengerDetails['firstName'])) {
                $passenger->first_name = $passengerDetails['firstName'];
                $createdName .= ",".$passengerDetails['firstName'];
            }
            $passenger->created_name = $createdName;
        }

        if (isset($passengerDetails["gender"]) && $passengerDetails["gender"]) {
            $passenger->gender = $passengerDetails['gender'];
        }
        if (isset($passengerDetails["dob"]) && $passengerDetails["dob"]){
            $passenger->dob = self::getFormattedDate($passengerDetails['dob'], true);
        }
        if (isset($passengerDetails["citizen"]) && $passengerDetails["citizen"]){
            if (strlen($passengerDetails['citizen']) == 3){
                if (isset($countryISOIDs[$passengerDetails['citizen']])){
                    $passenger->pob_country_id = $countryISOIDs[$passengerDetails['citizen']];
                }
            }
            else {
                if (isset($countryIDs[$passengerDetails['citizen']])){
                    $passenger->pob_country_id = $countryIDs[$passengerDetails['citizen']];
                }
            }
        }
        if (isset($passengerDetails["docNo"]) && $passengerDetails["docNo"]){
            $passenger->passport_num = $passengerDetails['docNo'];
        }
        if (isset($passengerDetails["docCountry"]) && $passengerDetails["docCountry"]){
            if (strlen($passengerDetails['docCountry']) == 3){
                if (isset($countryISOIDs[$passengerDetails['docCountry']])){
                    $passenger->passport_country_id = $countryISOIDs[$passengerDetails['docCountry']];
                }
            }
            else {
                if (isset($countryIDs[$passengerDetails['docCountry']])){
                    $passenger->passport_country_id = $countryIDs[$passengerDetails['docCountry']];
                }
            }
        }
        if (isset($passengerDetails["docDEX"]) && $passengerDetails["docDEX"]){
            $passenger->passport_dex = self::getFormattedDate($passengerDetails['docDEX']);
        }
        if (isset($passengerDetails["address"]) && $passengerDetails["address"]){
            $passenger->address = $passengerDetails['address'];
        }
        if (isset($passengerDetails["residentCountry"]) && $passengerDetails["residentCountry"]){
            if (strlen($passengerDetails['residentCountry']) == 3){
                if (isset($countryISOIDs[$passengerDetails['residentCountry']])){
                    $passenger->resident_country_id = $countryISOIDs[$passengerDetails['residentCountry']];
                }
            }
            else {
                if (isset($countryIDs[$passengerDetails['docCountry']])){
                    $passenger->resident_country_id = $countryIDs[$passengerDetails['residentCountry']];
                }
            }
        }

        $passenger->save();

        return $passenger;
    }



    public static function getFormattedDate($dmy, $dob = false)
    {
        if ($dmy && preg_match("/([0-9]{2})([A-Z]{3})([0-9]{2})/", $dmy, $matches))
        {
            list($str, $day, $month, $year) = $matches;
            $yearInt = (int)$year;
            if ($dob){
                $yearStr = ($yearInt > date("y")) ?  "19$year": "20$year";
            }
            else {
                $yearStr = ($yearInt > date("y")) ? "20$year" : "19$year";
            }
            return $yearStr."-".self::$months[$month]."-".$day;
        }
        return "";
    }

    public static function getFormattedDateMonth($dm)
    {
        if ($dm && preg_match("/([0-9]{2})([A-Z]{3})/", $dm, $matches))
        {
            list($str, $day, $month) = $matches;
            return date("Y")."-".self::$months[$month]."-".$day;
        }
        return null;
    }

    public static function findOrCreateInfPassenger($passengerDetails){
        $passenger = null;

        if ($passengerDetails['infLname']){
            $passenger = Passenger::where("last_name", $passengerDetails["infLname"]);

            if (isset($passengerDetails['infFname']) && $passengerDetails['infFname']){
                $passenger->where("first_name", $passengerDetails['infFname']);
            }

            if (isset($passengerDetails["infDOB"]) && $passengerDetails["infDOB"]){
                $passenger->where(function($sql) use ($passengerDetails){
                    $sql->where("dob", self::getFormattedDate($passengerDetails['infDOB'], true))
                        ->orWhereNull("dob");
                });
            }

            $passenger = $passenger->first();

            if (!$passenger) {
                $passenger = new Passenger();
                $passenger->last_name = $passengerDetails['infLname'];

                if (isset($passengerDetails['infFname']) && $passengerDetails['infFname']){
                    $passenger->first_name = $passengerDetails['infFname'];
                }
            }

            if (isset($passengerDetails["infDOB"]) && $passengerDetails["infDOB"]){
                $passenger->dob = self::getFormattedDate($passengerDetails['infDOB'], true);
            }

            $passenger->save();

        }

        return $passenger;
    }

    public static function findOrCreatePassengerOrder($passengerDetails, $infRecord = FALSE){
        $passengerOrder = $ticketNumber = null;

        $pnr = isset($passengerDetails["pnr"]) ? $passengerDetails["pnr"] : null;
        if ($infRecord){
            if(isset($passengerDetails['infTktNo'])) {
                $ticketNumber = $passengerDetails['infTktNo'];
            }
        } else {
            if (isset($passengerDetails['ticketNumber'])) {
                $ticketNumber = $passengerDetails['ticketNumber'];
            }
        }

        if (!$ticketNumber && !$pnr){
            return null;
        }

        if ($ticketNumber) {
            $passengerOrder = PassengerOrder::where("ticket_number", $ticketNumber)
                ->first();
        }

        if (!$passengerOrder){
            $passengerOrder = PassengerOrder::where("pnr", $pnr)
                ->first();
        }

        if (!$passengerOrder){
            $passengerOrder = new PassengerOrder();
        }

        $passengerOrder->ticket_number = $ticketNumber;
        $passengerOrder->pnr = $pnr;

        try {
            $passengerOrder->save();
        }
        catch (\Exception $exception){
            sendSimpleEmail($exception->getMessage(), "Passenger Order Creation Error");
            return null;
        }

        return $passengerOrder;
    }

    public static function findOrCreatePassengerFlight($passenger, $passengerOrder, $flight, $passengerDetails, $sentDate, $infRecord = null, $parentPassengerFlight = null, $act = FALSE, $source = PRL){
        if (! ($passenger && $flight) ){
            return null;
        }

        if ($infRecord){
            $coupon = isset($passengerDetails['infCpn']) ? $passengerDetails['infCpn'] : null;
        }
        else {
            $coupon = isset($passengerDetails['coupon']) ? $passengerDetails['coupon'] : null;
        }

        // Search for the record
        $passengerFlight = $act ? PassengerFlightAct::select() : PassengerFlight::select();

        $passengerFlight = $passengerFlight
                            ->where("passenger_id", $passenger->id)
                            ->where("flight_id", $flight->id)
                            ->get();

        if (count($passengerFlight) > 1){
            if ($coupon){
                $passengerFlight = $passengerFlight
                    ->where("passenger_id", $passenger->id)
                    ->where("flight_id", $flight->id)
                    ->where("coupon", $coupon)
                    ->first();
            }
        }
        elseif (count($passengerFlight) == 1){
            $passengerFlight = $passengerFlight->first();
        }
        else {
            $passengerFlight = null;
        }

        if (!$passengerFlight){
            $passengerFlight = $act ? new PassengerFlightAct() : new PassengerFlight();

            $passengerFlight->passenger_id = $passenger->id;
            $passengerFlight->flight_id = $flight->id;
            $passengerFlight->coupon = $coupon;
        }

        // Source
        if ($act) {
            switch ($source) {
                case PTM:
                    $passengerFlight->from_ptm = true;
                    break;

                case TPM:
                    $passengerFlight->from_tpm = true;
                    break;

                case PSM:
                    $passengerFlight->from_psm = true;
                    break;

                case PRL:
                    $passengerFlight->from_prl = true;
                    break;

                default:
                    break;
            }
        }

        // Update Coupon
        if ($coupon){
            $passengerFlight->coupon = $coupon;
        }

        // INF Record
        if ($infRecord){
            $passengerFlight->parent_id = $parentPassengerFlight ? $parentPassengerFlight->id : null;
            $passengerFlight->is_inf = true;
        }
        else {
            if (isset($passengerDetails['seqNo']) && $passengerDetails['seqNo']){
                $passengerFlight->seq_no = $passengerDetails['seqNo'];
            }

            if (isset($passengerDetails['groupTcp']) && $passengerDetails['groupTcp']){
                $passengerFlight->group_tcp = $passengerDetails['groupTcp'];
            }

            if (isset($passengerDetails['seat']) && $passengerDetails['seat']){
                $passengerFlight->seat_number = $passengerDetails['seat'];
            }

            // Frequent Flyer's Info
            $ff = self::findOrCreatePassengerFrequentFlyer($passenger, $passengerDetails);
            $passengerFlight->passenger_frequent_flyer_id = $ff ? $ff->id : null;

            // Extracted From PTM
            if (isset($passengerDetails["baggagePcs"]) && $passengerDetails["baggagePcs"]){
                $passengerFlight->baggage_pcs  = $passengerDetails['baggagePcs'];
            }
            if (isset($passengerDetails["baggageWeight"]) && $passengerDetails["baggageWeight"]){
                $passengerFlight->baggage_weight  = $passengerDetails['baggageWeight'];
            }

            if (isset($passengerDetails["tagNumber"]) && $passengerDetails["tagNumber"]){
                $passengerFlight->tag_number  = implode(";", $passengerDetails['tagNumber']);
            }

            if (isset($passengerDetails["ptmNameStr"]) && $passengerDetails["ptmNameStr"]){
                $passengerFlight->ptm_name_str  = $passengerDetails['ptmNameStr'];
            }
            if (isset($passengerDetails["ptmPax"]) && $passengerDetails["ptmPax"]){
                $passengerFlight->ptm_pax  = $passengerDetails['ptmPax'];
            }
            // END PTM

            // Only For ADULT Record
            if (isset($passengerDetails["xbagKgs"]) && $passengerDetails["xbagKgs"]){
                $passengerFlight->xbag_wt  = $passengerDetails['xbagKgs'];
            }
            if (isset($passengerDetails["xbagQty"]) && $passengerDetails["xbagQty"]){
                $passengerFlight->xbag_qty  = $passengerDetails['xbagQty'];
            }

            $info = [];

            if (isset($passengerDetails["xbagTxt1"]) && $passengerDetails["xbagTxt1"]){
                $info[] = $passengerDetails['xbagTxt1'];
            }
            if (isset($passengerDetails["xbagTxt2"]) && $passengerDetails["xbagTxt2"]){
                $info[] = $passengerDetails['xbagTxt2'];
            }
            $passengerFlight->xbag_info  = count($info) ? serialize($info) : null;

            // Outbound Info
            if (isset($passengerDetails['outbAirline'])){
                $outbound = [];
                $outbound['airline']               = $passengerDetails['outbAirline'];
                if (isset($passengerDetails['outbFltNo'])){
                    $outbound['flight_number']     = $passengerDetails['outbFltNo'];
                }
                if (isset($passengerDetails['outbRBD'])){
                    $outbound['rbd']               = $passengerDetails['outbRBD'];
                }
                if (isset($passengerDetails['outbDay'])){
                    $outbound['departure_date']    = date("Y-m-".$passengerDetails['outbDay'], strtotime($sentDate));
                }
                if (isset($passengerDetails['outbDeptTime'])){
                    $outbound['departure_time']    = date("H:i", strtotime($passengerDetails['outbDeptTime']));
                }
                if (isset($passengerDetails['outbDepAirline'])){
                    $outbound['departure_airline'] = $passengerDetails['outbDepAirline'];
                }
                if (isset($passengerDetails['outbDest'])){
                    $outbound['destination']       = $passengerDetails['outbDest'];
                }

                // Outbound - 2  Info
                if (isset($passengerDetails['outb2Airline'])){
                    $outbound['airline2']               = $passengerDetails['outb2Airline'];
                    if (isset($passengerDetails['outb2FltNo'])){
                        $outbound['flight_number2']     = $passengerDetails['outb2FltNo'];
                    }
                    if (isset($passengerDetails['outb2RBD'])){
                        $outbound['rbd2']               = $passengerDetails['outb2RBD'];
                    }
                    if (isset($passengerDetails['outb2Day'])){
                        $outbound['departure_date2']    = date("Y-m-".$passengerDetails['outb2Day'], strtotime($sentDate));
                    }
                    if (isset($passengerDetails['outb2DeptTime'])){
                        $outbound['departure_time2']    = date("H:i", strtotime($passengerDetails['outb2DeptTime']));
                    }
                    if (isset($passengerDetails['outb2DepAirline'])){
                        $outbound['departure_airline2'] = $passengerDetails['outb2DepAirline'];
                    }
                    if (isset($passengerDetails['outb2Dest'])){
                        $outbound['destination2']       = $passengerDetails['outb2Dest'];
                    }
                }

                $passengerFlight->outbound = serialize($outbound);
            }

            // Inbound Info
            if (isset($passengerDetails['inbAirline'])){
                $inbound = [];
                $inbound['airline']           = $passengerDetails['inbAirline'];
                if (isset($passengerDetails['inbFltNo'])){
                    $inbound['flight_number']     = $passengerDetails['inbFltNo'];
                }

                if (isset($passengerDetails['inbRBD'])){
                    $inbound['rbd']               = $passengerDetails['inbRBD'];
                }
                if (isset($passengerDetails['inbDay'])){
                    $inbound['arrival_date']      = date("Y-m-".$passengerDetails['inbDay'], strtotime($sentDate));
                }
                if (isset($passengerDetails['inbArrTime'])){
                    $inbound['arrival_time']      = date("H:i", strtotime($passengerDetails['inbArrTime']));
                }
                if (isset($passengerDetails['inbArrAirline'])){
                    $inbound['arrival_airline']   = $passengerDetails['inbArrAirline'];
                }
                if (isset($passengerDetails['inbOrig'])){
                    $inbound['origin']            = $passengerDetails['inbOrig'];
                }
                $passengerFlight->inbound = serialize($inbound);
            }

            // SSR
            if (isset($passengerDetails["ssr"]) && $passengerDetails["ssr"]){
                $ssr = $passengerDetails['ssr'];

                if (isset($passengerDetails["ssr2"]) && $passengerDetails["ssr2"]){
                    $ssr .= ";".$passengerDetails['ssr2'];
                }

                $passengerFlight->ssr  = $ssr;

                if (isset($passengerDetails["ssr_info"]) && $passengerDetails["ssr_info"]){
                    $passengerFlight->ssr_info = $passengerDetails['ssr_info'];
                }
            }

            // Applies only for PNL
            if (!$act) {
                // SPML
                if (isset($passengerDetails["spml"]) && $passengerDetails["spml"]) {
                    $passengerFlight->spml = $passengerDetails['spml'];
                }

                // EPAY
                if (isset($passengerDetails["epay_kk"]) && $passengerDetails["epay_kk"]) {
                    $passengerFlight->epay_kk = $passengerDetails['epay_kk'];
                }
                if (isset($passengerDetails["epay_cc"]) && $passengerDetails["epay_cc"]) {
                    $passengerFlight->epay_cc = $passengerDetails['epay_cc'];
                }

                // CODE-SHARE FLIGHT
                if (isset($passengerDetails['codeShareDepAirline'])){
                    $codeShare = [];
                    $codeShare['airline']               = $passengerDetails['codeShareDepAirline'];
                    if (isset($passengerDetails['codeShareFltNo'])){
                        $codeShare['flight_number']     = $passengerDetails['codeShareFltNo'];
                    }
                    if (isset($passengerDetails['codeShareRBD'])){
                        $codeShare['rbd']               = $passengerDetails['codeShareRBD'];
                    }
                    if (isset($passengerDetails['codeShareDay'])){
                        $codeShare['departure_date']    = date("Y-m-".$passengerDetails['codeShareDay'], strtotime($sentDate));
                    }
                    if (isset($passengerDetails['codeShareOrig'])){
                        $codeShare['origin']       = $passengerDetails['codeShareOrig'];
                    }
                    if (isset($passengerDetails['codeShareDest'])){
                        $codeShare['destination']       = $passengerDetails['codeShareDest'];
                    }
                    $passengerFlight->codeshare_flight = serialize($codeShare);
                }
            }
        }

        if (isset($passengerDetails['adlAction']) && $passengerDetails['adlAction'] == self::DEL){
            $passengerFlight->deleted_at = date("Y-m-d H:i:s");
        }

        $passengerFlight->flight_number_id = $flight->flight_number_id;
        if ($passengerOrder){
            $passengerFlight->passenger_order_id = $passengerOrder->id;
        }
        $passengerFlight->departure = $flight->ptd ? $flight->ptd : $flight->std;
        $passengerFlight->departure_date = $flight->ptd ? date("Y-m-d", strtotime($flight->ptd)) : date("Y-m-d", strtotime($flight->std));
        $passengerFlight->save();

        return $passengerFlight;
    }

    public static function getPassengerGender(&$passengerDetails){
        $passengerDetails['gender'] = null;
        if ($passengerDetails['firstName']){
            $fName = trim($passengerDetails['firstName']);
            $found = true;
            switch($fName){
                case MRS:
                case MS:
                case MISS:
                    $passengerDetails['gender'] = FEMALE;
                    $passengerDetails['firstName'] = "";
                    break;

                case MR:
                    $passengerDetails['gender'] = MALE;
                    $passengerDetails['firstName'] = "";
                    break;

                case CHD:
                    $passengerDetails['firstName'] = "";
                    break;

                default:
                    $passengerDetails['firstName'] = $fName;
                    $found = false;
                    break;
            }

            if (!$found){
                preg_match('/([A-Z]+)\s*(MISS|MS|MRS|MR|DR|CHD)\b/', $passengerDetails['firstName'], $match);
                if ($match && isset($match[2])){
                    switch($match[2]){
                        case MRS:
                        case MS:
                        case MISS:
                            $passengerDetails['gender'] = FEMALE;
                            $passengerDetails['firstName'] = $match[1];
                            break;

                        case MR:
                            $passengerDetails['gender'] = MALE;
                            $passengerDetails['firstName'] = $match[1];
                            break;

                        case CHD:
                        default:
                            $passengerDetails['firstName'] = $match[1];
                            break;
                    }
                }
            }
        }
    }

    public static function getGender($gender){
        switch($gender){
            case MRS:
            case MS:
            case MISS:
                return FEMALE;
                break;

            case MR:
                return MALE;
                break;

            default:
                break;
        }
        return null;
    }

    public static function getPnlArrayRecord(&$record, $matches){
        // INFANT DETAILS
        if (isset($matches[1] ) && $matches[1] ){ $record['infDOB'] 		= $matches[1] ; }
        if (isset($matches[2] ) && $matches[2] ){ $record['infLname'] 		= $matches[2] ; }
        if (isset($matches[3] ) && $matches[3] ){ $record['infFname'] 		= $matches[3] ; }
        if (isset($matches[4] ) && $matches[4] ){ $record['infTktNo'] 		= $matches[4] ; }
        if (isset($matches[5] ) && $matches[5] ){ $record['infCpn'] 		= $matches[5] ; }
        if (isset($matches[6] ) && $matches[6] ){ $record['infParentLName'] = $matches[6] ; }
        if (isset($matches[7] ) && $matches[7] ){ $record['infParentFName'] = $matches[7] ; }

        // TICKET NO DETAILS
        if (isset($matches[8] ) && $matches[8] ){ $record['ticketNumber'] 	    = $matches[8] ; }
        if (isset($matches[9] ) && $matches[9] ){ $record['coupon'] 		    = $matches[9] ; }
        if (isset($matches[10]) && $matches[10]){ $record['ticketNumberLName'] 	= $matches[10] ; }
        if (isset($matches[11]) && $matches[11]){ $record['ticketNumberFName'] 	= $matches[11] ; }

        // FQTV
        if (isset($matches[12]) && $matches[12]){ $record['ffAirline']      = trim($matches[12]) ; }
        if (isset($matches[13]) && $matches[13]){ $record['ffNo'] 		    = $matches[13] ; }
        if (isset($matches[14]) && $matches[14]){ $record['seqNo'] 			= $matches[14]; }

        // SEAT
        if (isset($matches[15]) && $matches[15]){ $record['seat'] 			= $matches[15]; }
        if (isset($matches[16]) && $matches[16]){ $record['seatLName'] 		= $matches[16]; }
        if (isset($matches[17]) && $matches[17]){ $record['seatFName'] 		= $matches[17]; }

        // PNR
        if (isset($matches[18]) && $matches[18]){ $record['pnr'] 			= $matches[18]; }

        // OUTBOUND FLIGHT
        if (isset($matches[19]) && $matches[19]){ $record['outbAirline'] 	= $matches[19]; }
        if (isset($matches[20]) && $matches[20]){ $record['outbFltNo'] 		= $matches[20]; }
        if (isset($matches[21]) && $matches[21]){ $record['outbRBD'] 		= $matches[21]; }
        if (isset($matches[22]) && $matches[22]){ $record['outbDay'] 		= $matches[22]; }
        if (isset($matches[23]) && $matches[23]){ $record['outbDest'] 		= $matches[23]; }
        if (isset($matches[24]) && $matches[24]){ $record['outbDeptTime'] 	= $matches[24]; }
        if (isset($matches[25]) && $matches[25]){ $record['outbDepAirline']	= $matches[25]; }

        // INBOUND FLIGHT
        if (isset($matches[26]) && $matches[26]){ $record['inbAirline'] 	= $matches[26]; }
        if (isset($matches[27]) && $matches[27]){ $record['inbFltNo'] 		= $matches[27]; }
        if (isset($matches[28]) && $matches[28]){ $record['inbRBD'] 		= $matches[28]; }
        if (isset($matches[29]) && $matches[29]){ $record['inbDay'] 		= $matches[29]; }
        if (isset($matches[30]) && $matches[30]){ $record['inbOrig'] 		= $matches[30]; }
        if (isset($matches[31]) && $matches[31]){ $record['inbArrTime']		= $matches[31]; }
        if (isset($matches[32]) && $matches[32]){ $record['inbArrAirline']	= $matches[32]; }

        // XBAG 1 OPTION
        if (isset($matches[33]) && $matches[33]){ $record['xbagQty'] 		= $matches[33]; }
        if (isset($matches[34]) && $matches[34]){ $record['xbagTxt1'] 		= $matches[34]; }
        if (isset($matches[35]) && $matches[35]){ $record['xbagKgs'] 		= $matches[35]; }
        if (isset($matches[36]) && $matches[36]){ $record['xbagTxt2'] 		= $matches[36]; }

        // XBAG 2 OPTION
        if (isset($matches[37]) && $matches[37]){ $record['xbagQty'] 		= $matches[37]; }
        if (isset($matches[38]) && $matches[38]){ $record['xbagKgs'] 		= $matches[38]; }
        if (isset($matches[39]) && $matches[39]){ $record['xbagLName'] 		= $matches[39]; }
        if (isset($matches[40]) && $matches[40]){ $record['xbagFName'] 		= $matches[40]; }

        // DOCS
        if (isset($matches[41]) && $matches[41]){ $record['docSeries'] 		= $matches[41]; }
        if (isset($matches[42]) && $matches[42]){ $record['docCountry'] 	= $matches[42]; }
        if (isset($matches[43]) && $matches[43]){ $record['docNo'] 			= $matches[43]; }
        if (isset($matches[44]) && $matches[44]){ $record['citizen'] 		= $matches[44]; }
        if (isset($matches[45]) && $matches[45]){ $record['dob']			= $matches[45]; }
        if (isset($matches[46]) && $matches[46]){ $record['gender'] 		= $matches[46]; }
        if (isset($matches[47]) && $matches[47]){ $record['docDEX'] 		= $matches[47]; }
        if (isset($matches[48]) && $matches[48]){ $record['docLName'] 		= $matches[48]; }
        if (isset($matches[49]) && $matches[49]){ $record['docFName'] 		= $matches[49]; }

        // SPECIAL MEAL
        if (isset($matches[50]) && $matches[50]){ $record['spml'] 		    = $matches[50]; }
        if (isset($matches[51]) && $matches[51]){ $record['spmlLName'] 		= $matches[51]; }
        if (isset($matches[52]) && $matches[52]){ $record['spmlFName'] 		= $matches[52]; }

        if (isset($matches[53]) && $matches[53]){ $record['epay_kk'] 		= $matches[53]; }
        if (isset($matches[54]) && $matches[54]){ $record['epay_cc'] 		= $matches[54]; }

        // OUTBOUND-2 FLIGHT
        if (isset($matches[55]) && $matches[55]){ $record['outb2Airline'] 	= $matches[55]; }
        if (isset($matches[56]) && $matches[56]){ $record['outb2FltNo'] 	= $matches[56]; }
        if (isset($matches[57]) && $matches[57]){ $record['outb2RBD'] 		= $matches[57]; }
        if (isset($matches[58]) && $matches[58]){ $record['outb2Day'] 		= $matches[58]; }
        if (isset($matches[59]) && $matches[59]){ $record['outb2Dest'] 		= $matches[59]; }
        if (isset($matches[60]) && $matches[60]){ $record['outb2DeptTime'] 	= $matches[60]; }
        if (isset($matches[61]) && $matches[61]){ $record['outb2DepAirline']= $matches[61]; }

        // CODESHARE FLIGHT
        if (isset($matches[62]) && $matches[62]){ $record['codeShareDepAirline']= $matches[62]; }
        if (isset($matches[63]) && $matches[63]){ $record['codeShareFltNo'] 	= $matches[63]; }
        if (isset($matches[64]) && $matches[64]){ $record['codeShareRBD'] 		= $matches[64]; }
        if (isset($matches[65]) && $matches[65]){ $record['codeShareDay'] 		= $matches[65]; }
        if (isset($matches[66]) && $matches[66]){ $record['codeShareOrig'] 		= $matches[66]; }
        if (isset($matches[67]) && $matches[67]){ $record['codeShareDest'] 		= $matches[67]; }
    }

    public static function getPrlArrayRecord(&$record, $matches){

        if (isset($matches[1] ) && $matches[1] ){ $record['infDOB'] 		= $matches[1] ; }
        if (isset($matches[2] ) && $matches[2] ){ $record['infLname'] 		= $matches[2] ; }
        if (isset($matches[3] ) && $matches[3] ){ $record['infFname'] 		= $matches[3] ; }
        if (isset($matches[4] ) && $matches[4] ){ $record['infTktNo'] 		= $matches[4] ; }
        if (isset($matches[5] ) && $matches[5] ){ $record['infCpn'] 		= $matches[5] ; }

        // TICKET NO
        if (isset($matches[6] ) && $matches[6] ){ $record['ticketNumber'] 	= $matches[6] ; }
        if (isset($matches[7] ) && $matches[7] ){ $record['coupon'] 		= $matches[7] ; }
        // DONE

        // FQTV
        if (isset($matches[8] ) && $matches[8] ){ $record['ffAirline'] 		= trim($matches[8]) ; }
        if (isset($matches[9] ) && $matches[9] ){ $record['ffNo'] 			= $matches[9] ; }
        // DONE

        if (isset($matches[10]) && $matches[10]){ $record['seqNo'] 			= $matches[10]; }

        // SEAT
        if (isset($matches[11]) && $matches[11]){ $record['seat'] 			= $matches[11]; }
        // PNR
        if (isset($matches[12]) && $matches[12]){ $record['pnr'] 			= $matches[12]; }
        // BAG PCS
        if (isset($matches[13]) && $matches[13]){ $record['baggagePcs'] 	= $matches[13]; }
        // BAG WT
        if (isset($matches[14]) && $matches[14]){ $record['baggageWeight'] 	= $matches[14]; }
        // BAG TAGS
        if (isset($matches[15]) && $matches[15]){
            if (!isset($record['tagNumber'])){
                $record['tagNumber'] = [];
            }
            $record['tagNumber'][]	= $matches[15];
        }
        // DONE

        // OUTBOUND
        if (isset($matches[16]) && $matches[16]){ $record['outbAirline'] 	= $matches[16]; }
        if (isset($matches[17]) && $matches[17]){ $record['outbFltNo'] 		= $matches[17]; }
        if (isset($matches[18]) && $matches[18]){ $record['outbRBD'] 		= $matches[18]; }
        if (isset($matches[19]) && $matches[19]){ $record['outbDay'] 		= $matches[19]; }
        if (isset($matches[20]) && $matches[20]){ $record['outbDest'] 		= $matches[20]; }
        // DONE

        if (isset($matches[21]) && $matches[21]){ $record['outbDeptTime'] 	= $matches[21]; }
        if (isset($matches[22]) && $matches[22]){ $record['outbDepAirline']	= $matches[22]; }

        // INBOUND
        if (isset($matches[23]) && $matches[23]){ $record['inbAirline'] 	= $matches[23]; }
        if (isset($matches[24]) && $matches[24]){ $record['inbFltNo'] 		= $matches[24]; }
        if (isset($matches[25]) && $matches[25]){ $record['inbRBD'] 		= $matches[25]; }
        if (isset($matches[26]) && $matches[26]){ $record['inbDay'] 		= $matches[26]; }
        if (isset($matches[27]) && $matches[27]){ $record['inbOrig'] 		= $matches[27]; }
        // DONE

        if (isset($matches[28]) && $matches[28]){ $record['inbArrTime']		= $matches[28]; }
        if (isset($matches[29]) && $matches[29]){ $record['inbArrAirline']	= $matches[29]; }

        if (isset($matches[30]) && $matches[30]){ $record['xbagQty'] 		= $matches[30]; }
        if (isset($matches[31]) && $matches[31]){ $record['xbagTxt1'] 		= $matches[31]; }
        if (isset($matches[32]) && $matches[32]){ $record['xbagKgs'] 		= $matches[32]; }
        if (isset($matches[33]) && $matches[33]){ $record['xbagTxt2'] 		= $matches[33]; }

        // BELOW DONE
        if (isset($matches[34]) && $matches[34]){ $record['docSeries'] 		= $matches[34]; }
        if (isset($matches[35]) && $matches[35]){ $record['docCountry'] 	= $matches[35]; }
        if (isset($matches[36]) && $matches[36]){ $record['docNo'] 			= $matches[36]; }
        if (isset($matches[37]) && $matches[37]){ $record['citizen'] 		= $matches[37]; }
        if (isset($matches[38]) && $matches[38]){ $record['dob']			= $matches[38]; }
        if (isset($matches[39]) && $matches[39]){ $record['gender'] 		= $matches[39]; }
        if (isset($matches[40]) && $matches[40]){ $record['docDEX'] 		= $matches[40]; }
        if (isset($matches[41]) && $matches[41]){ $record['docLastName'] 	= $matches[41]; }
        if (isset($matches[42]) && $matches[42]){ $record['docFirstName'] 	= $matches[42]; }

        if (isset($matches[44]) && $matches[44]){ $record['residentCountry'] 	= $matches[44]; }
        if (isset($matches[45]) && $matches[45]){ $record['address'] 	        = $matches[45]; }
    }


    public static function trimNameAndGetGender($name){
        $gen = $isChild = $isInf = null;
        $fName = trim($name);
        if (ends_with($fName, MRS)){
            $gen = FEMALE;
            $fName = substr($fName, 0, -3);
        }
        if (ends_with($fName, MISS)){
            $gen = FEMALE;
            $fName = substr($fName, 0, -4);
        }
        if (ends_with($fName, MS)){
            $gen = FEMALE;
            $fName = substr($fName, 0, -2);
        }
        else if (ends_with($fName, "DRMR")){
            $gen = MALE;
            $fName = substr($fName, 0, -4);
        }
        else if (ends_with($fName, MR)){
            $gen = MALE;
            $fName = substr($fName, 0, -2);
        }
        else if (ends_with($fName, DR)){
            $fName = substr($fName, 0, -2);
        }
        else if (ends_with($fName, MSTR)){
            $gen = MALE;
            $fName = substr($fName, 0, -4);
        }
        else if (ends_with($fName, CHD)){
            $fName = substr($fName, 0, -3);
            $isChild = true;
        }
        else if (ends_with($fName, INFT)){
            $isInf = true;
            $fName = substr($fName, 0, -4);
        }

        $fName = trim($fName);

        return[ $fName, $gen, $isChild ,$isInf ];
    }
}
