<?php


namespace App\Classes\FlightStaff;


use App\Models\AirlineService;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\StaffService;

class StaffFilterWatch
{
    protected $loadAllStationsStaff;
    protected $includeStaffData;

    protected $handlingStations;
    protected $occStaffOnly;
    protected $airportID;

    protected $currentServiceType;
    protected $currentFlightServiceType;
    // OUTPUTS

    protected $rosteredStaffServices = [];

    protected $rosteredStaffUserIDs = null;

    public function __construct($airportID, $handlingStations = null, $loadAllStationsStaff = false, $includeStaffData = false, $occStaffOnly = false)
    {
        $this->airportID = $airportID;
        $this->occStaffOnly = $occStaffOnly;
        $this->includeStaffData = $includeStaffData;
        $this->loadAllStationsStaff = $loadAllStationsStaff;

        if (!$handlingStations){
            $handlingStations = Airport::getHandlingStationsIdAndObject();
        }
        $this->handlingStations = $handlingStations;
    }



    function getRosteredStaffService($flight){
        $serviceType = $this->getServiceType($flight);

        if ($serviceType != ANOTHER_AIRPORT_SERVICE){
            return $serviceType;
        }

        if (count($this->rosteredStaffServices) == 1){
            return $this->rosteredStaffServices[0];
        }

//        debug($serviceType);

        return null;
    }

    function getServiceType($flight){

        $flightNumber = $flight->flightNumber;
        $departureAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flightNumber->departure_airport_id;
        $arrivalAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flightNumber->arrival_airport_id;
        $this->currentFlightServiceType = null;

        // If AP
        if ($this->airportID)
        {
            if ($this->airportID == $departureAirportID &&
                array_key_exists($departureAirportID, $this->handlingStations))
            {
                if (array_key_exists($arrivalAirportID, $this->handlingStations))
                {
                    $this->currentFlightServiceType = FULL_SERVICE;
                }

                return DEPARTURE_SERVICE;
            }
            elseif ($this->airportID == $arrivalAirportID &&
                array_key_exists($arrivalAirportID, $this->handlingStations))
            {
                if (array_key_exists($departureAirportID, $this->handlingStations))
                {
                    $this->currentFlightServiceType = FULL_SERVICE;
                }

                return ARRIVAL_SERVICE;
            }
            else {
                return ANOTHER_AIRPORT_SERVICE;
            }
        }

        if (array_key_exists($departureAirportID, $this->handlingStations)
            && array_key_exists($arrivalAirportID, $this->handlingStations))
        {
            return FULL_SERVICE;
        }
        elseif (array_key_exists($departureAirportID, $this->handlingStations))
        {
            return DEPARTURE_SERVICE;
        }
        else if (array_key_exists($arrivalAirportID, $this->handlingStations))
        {
            return ARRIVAL_SERVICE;
        }

        return null;
    }

    function getServiceTypeWithVariables($flight){
        $flightNumber = $flight->flightNumber;
        $departureAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flightNumber->departure_airport_id;
        $arrivalAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flightNumber->arrival_airport_id;
        $station = $serviceType = $type = null;

        if ($this->airportID)
        {
            if ($this->airportID == $departureAirportID &&
                array_key_exists($departureAirportID, $this->handlingStations))
            {
                $station = $this->handlingStations[$this->airportID];
                $serviceType = DEPARTURE_SERVICE;
            }
            elseif ($this->airportID == $arrivalAirportID &&
                array_key_exists($arrivalAirportID, $this->handlingStations))
            {
                $station = $this->handlingStations[$this->airportID];
                $serviceType = ARRIVAL_SERVICE;
            }

            if ($serviceType){
                return [$serviceType, $station];
            }
        }

        if (array_key_exists($departureAirportID, $this->handlingStations)
            && array_key_exists($arrivalAirportID, $this->handlingStations))
        {
            $station = null;
            $type = FULL_SERVICE;
        }
        elseif (array_key_exists($departureAirportID, $this->handlingStations))
        {
            $station = $this->handlingStations[$departureAirportID];
            $type = DEPARTURE_SERVICE;
        }
        else if (array_key_exists($arrivalAirportID, $this->handlingStations))
        {
            $station = $this->handlingStations[$arrivalAirportID];
            $type = ARRIVAL_SERVICE;
        }


        return [$type, $station];
    }

    function getStaffData($flights, $asArray = false){
        foreach ($flights as &$flight){
            $flight->serviceData = $this->getFlightServiceData($flight);
        }

        if ($asArray){
            return $flights->toArray();
        }

        return $flights;
    }

    function getRosterFlightStaffData($flights){
        $data = [];

        foreach ($flights as &$flight){
            $result = $this->getFlightServiceData($flight);

            if (!$result || ($result && !count($result["services"]))){
                continue;
            }

            $data[$flight->id] = $result;

        }

        return $data;
    }

    function setRosteredStaffUsers($flight){
        if ($this->includeStaffData){
            $this->rosteredStaffUserIDs = Flight::getFlightStaffUsersObject($flight);
        }
    }

    function getFlightServiceData($flight){
        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;

        $this->setRosteredStaffUsers($flight);

        $this->currentServiceType = $this->getServiceType($flight);

        switch ($this->currentServiceType){
            case ANOTHER_AIRPORT_SERVICE:
                // DEP
                $eligibleServices = $this->getDepartureEligibleServices($flight);

                $depArrTime = getFlightDepartureInitialDate($flight);
                $flightType = "<span class='label label-warning'>DEP</span>";
                $typeText = "DEP";

                if (!count($eligibleServices)) {
                    // ARR
                    $eligibleServices = $this->getArrivalEligibleServices($flight);

                    $depArrTime = getFlightArrivalInitialDate($flight);
                    $flightType = "<span class='label label-warning'>ARR</span>";
                    $typeText = "ARR";
                }

                break;


            case FULL_SERVICE:
                // DEP
                $eligibleServices = $this->getDepartureEligibleServices($flight);

                // ARR
                $eligibleServices2 = $this->getArrivalEligibleServices($flight);
                foreach ($eligibleServices2 as $each){
                    $eligibleServices[] = $each;
                }

                $typeText = "DEP-ARR";
                $depArrTime = getFlightDepartureInitialDate($flight);
                $flightType = "<span class='label label-warning'>DEP-ARR</span>";
                break;

            case DEPARTURE_SERVICE:
                $eligibleServices = $this->getDepartureEligibleServices($flight);

                // ARR
                if ($this->currentFlightServiceType == FULL_SERVICE){
                    $eligibleServices2 = $this->getArrivalEligibleServices($flight);
                    foreach ($eligibleServices2 as $each){
                        $eligibleServices[] = $each;
                    }
                }

                $typeText = "DEP";
                $depArrTime = getFlightDepartureInitialDate($flight);
                $flightType = "<span class='label label-success'>DEP</span>";
                break;

            case ARRIVAL_SERVICE:
                $eligibleServices = $this->getArrivalEligibleServices($flight);

                // ARR
                if ($this->currentFlightServiceType == FULL_SERVICE){
                    $eligibleServices2 = $this->getDepartureEligibleServices($flight);
                    foreach ($eligibleServices2 as $each){
                        $eligibleServices[] = $each;
                    }
                }

                $typeText = "ARR";
                $depArrTime = getFlightArrivalInitialDate($flight);
                $flightType = "<span class='label label-primary'>ARR</span>";
                break;

            default:
                $depArrTime = null;
                $typeText = "-";
                $flightType = "-";
                $eligibleServices = [];
                break;
        }

        $sector = getFlightSector($flight);

        return [
            'id'            => $flight->id,
            'flight'        => "/{$typeText}/ ".$airline->iata."-".$flightNumber->flight_number."(".$sector.") ".baseDateFormat($depArrTime, true, "-", true),
            'serviceType'   => $this->currentServiceType,
            'flightType'    => $flightType,
            "depArrTime"    => $depArrTime,
            'services'      => $eligibleServices,
            'station'       => "",
        ];
    }
    
    protected function getDepartureEligibleServices($flight){
        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;
        $departureAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flightNumber->departure_airport_id;

        $airlineServices = AirlineService::getServices($airline->id, $departureAirportID, $this->occStaffOnly);
        $depArrTime = getFlightDepartureInitialDate($flight);
        return $this->getEligibleServices($flight, $departureAirportID, $airlineServices, DEPARTURE_SERVICE, $depArrTime);
    }

    protected function getArrivalEligibleServices($flight){
        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;
        $arrivalAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flightNumber->arrival_airport_id;

        $airlineServices = AirlineService::getServices($airline->id, $arrivalAirportID, $this->occStaffOnly);
        $depArrTime = getFlightArrivalInitialDate($flight);
        return $this->getEligibleServices($flight, $arrivalAirportID, $airlineServices, ARRIVAL_SERVICE, $depArrTime);
    }


    protected function getEligibleServices($flight, $stationID, $airlineServices, $serviceType, $depArrTime){
        $eligibleServices = [];

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;

        /* Is declared above - stations are not needed
        $stations[] = $stationID;
        if ($this->airportID && !in_array($this->airportID, $stations)){
            $stations[] = $this->airportID;
        }

        $rosteredStaffUserIDs = Flight::getFlightStaffUsersObject($flight, $stations);

        debug("Rostered Staff Count:".($rosteredStaffUserIDs ? count($rosteredStaffUserIDs) : 0));
        */

        if ($this->currentServiceType == ANOTHER_AIRPORT_SERVICE && !count($this->rosteredStaffUserIDs)){
            return $eligibleServices;
        }

        // Perchance here we have to get Flights Manual SLAs list and it should be similar to Airline Service

        foreach ($airlineServices as $j => $airlineService) {

            if (!checkSLAEffectivePeriod($airlineService, $depArrTime)){
                continue;
            }

            // SKIP if no appropriate service found
            if ($serviceType == ARRIVAL_SERVICE){
                $service_type = ARRIVAL_SERVICE;

                if ( ($airlineService->service_timings && !$airlineService->service->arrival_service)
                    || (!$airlineService->service_timings && !$airlineService->arrival_service)
                ){
                    continue;
                }
            }
            else {
                $service_type = DEPARTURE_SERVICE;
                if ( ($airlineService->service_timings && !$airlineService->service->departure_service)
                    || (!$airlineService->service_timings && !$airlineService->departure_service)
                ){
                    if (($airlineService->service_timings && !$airlineService->service->turnaround_service)
                        || (!$airlineService->service_timings && !$airlineService->turnaround_service)){
                        continue;
                    }
                    else {
                        if (staffFlightIsInactiveByID($flight->parent_id)){
                            continue;
                        }
                        else {
                            $service_type = TURNAROUND_SERVICE;
                        }
                    }
                }
            }
            // End

            // TYPE
            $airlineService->another_airport_type = $this->currentServiceType == ANOTHER_AIRPORT_SERVICE;

            if ($this->loadAllStationsStaff || $this->includeStaffData){
                $eligibleStaff = StaffService::getStaffWithLocation($airline->id, $airlineService, $stationID);
            }
            else {
                $eligibleStaff = StaffService::getStaff($stationID, $airline->id, $airlineService);
            }

            $rosteredStaff = [];
            if ($this->rosteredStaffUserIDs && isset($this->rosteredStaffUserIDs[$airlineService->service->abbr])
                && count($this->rosteredStaffUserIDs[$airlineService->service->abbr])){

                foreach ($eligibleStaff as $k => $flightStaff) {
                    if (isset($this->rosteredStaffUserIDs[$airlineService->service->abbr][$flightStaff->user_id])){

                        if ($this->airportID) {
                            if ($this->currentServiceType == ANOTHER_AIRPORT_SERVICE
                                && $flightStaff->user->location->airport_id != $this->airportID) {
//                            debug("STAFF AP: {$flightStaff->user->location->airport_id} / STATION ID: {$this->airportID}" );
                                continue;
                            }

                            if ($this->currentFlightServiceType == FULL_SERVICE
                                && $flightStaff->user->location->airport_id != $this->airportID
                                && $this->rosteredStaffUserIDs[$airlineService->service->abbr][$flightStaff->user_id]->service_type != $this->currentServiceType) {
//                            debug("STAFF AP: {$flightStaff->user->location->airport_id} / Service TYPE: {$flightStaff->service_type} / STATION ID: {$this->airportID}" );
                                continue;
                            }
                        }

                        $eligibleStaff[$k]->selected = true;

                        $rosteredStaff[] = $this->rosteredStaffUserIDs[$airlineService->service->abbr][$flightStaff->user_id];

                        // Services of Rostered Staff
                        if (!in_array($service_type, $this->rosteredStaffServices)){
                            $this->rosteredStaffServices[] = $service_type;
                        }
                    }
                }
            }

            // Skip other station SLAs if
            if ($this->airportID && $this->includeStaffData && !count($rosteredStaff)){
                if ($this->airportID != $stationID){
                    continue;
                }
            }


            /**
             * Prepared Data
             */
            if ($airlineService->service_timings == 1) {
                if ($airlineService->service[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService->service;
            }
            else {
                if ($airlineService[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService;
            }

            switch ($service_type){
                case ARRIVAL_SERVICE:
                    $itemServiceType = ARRIVAL_SERVICE;
                    $req = "arr_staff_req";
                    $min = "arr_staff_min";
                    $arr = getFlightArrivalInitialDatePTA($flight);
                    $plannedReport = date("Y-m-d H:i:s", strtotime($arr) - (60 * $obj->arr_report_time));
                    $plannedRelease = date("Y-m-d H:i:s", strtotime($arr) + (60 * $obj->arr_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case DEPARTURE_SERVICE:
                    $itemServiceType = DEPARTURE_SERVICE;
                    $req = "dep_staff_req";
                    $min = "dep_staff_min";
                    $dep = getFlightDepartureInitialDatePTD($flight);
                    $plannedReport = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case TURNAROUND_SERVICE:
                    $itemServiceType = TURNAROUND_SERVICE;
                    $req = "turn_staff_req";
                    $min = "turn_staff_min";

                    $arr = getFlightArrivalInitialDatePTA(getParentFlight($flight));
                    $dep = getFlightDepartureInitialDatePTD($flight);

                    $plannedReport = date("Y-m-d H:i", strtotime($arr) - (60 * $obj->turn_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->turn_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                default:
                    $itemServiceType = $plannedReport = $plannedRelease = $duration = null;
                    break;
            }

            $sla = $obj->service ? $obj->service->abbr : $obj->abbr;

            $airlineServices[$j]->sla = $sla;
            $airlineServices[$j]->req = $airlineService->service_timings ? $airlineService->service->{$req} : $airlineService->{$req};
            $airlineServices[$j]->report_time = $plannedReport;
            $airlineServices[$j]->release_time = $plannedRelease;
            $airlineServices[$j]->duration = $duration;
            $airlineServices[$j]->rosteredStaff = $rosteredStaff;
            $airlineServices[$j]->eligibleStaff = $eligibleStaff;
            $airlineServices[$j]->serviceType = $itemServiceType;

            // Add to Services Array
            $eligibleServices[] = $airlineServices[$j];
        }

        return $eligibleServices;
    }

    public function getFlightAirlineService($flight, &$airlineService, $serviceType){

        if (!$flight || !$flight->flightNumber || !$flight->flightNumber->airline){
            return null;
        }

        $station = $depArrTime = null;

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;
        $departureAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flightNumber->departure_airport_id;
        $arrivalAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flightNumber->arrival_airport_id;

//        $airportId = !$airportId ? self::getFlightNumbersHandlingStation($this->handlingStations, $flightNumber) : $airportId;
        switch ($serviceType){
            case ARRIVAL_SERVICE:
                $station = $this->handlingStations[$arrivalAirportID];

                if ( ($airlineService->service_timings && !$airlineService->service->arrival_service)
                    || (!$airlineService->service_timings && !$airlineService->arrival_service)
                ){
                    return null;
                }

                $depArrTime = getFlightArrivalInitialDate($flight);
                break;

            case DEPARTURE_SERVICE:
                $station = $this->handlingStations[$departureAirportID];

                if ( ($airlineService->service_timings && !$airlineService->service->departure_service)
                    || (!$airlineService->service_timings && !$airlineService->departure_service)
                ){
                    return null;
                }

                $depArrTime = getFlightDepartureInitialDate($flight);
                break;

            case TURNAROUND_SERVICE:
                $station = $this->handlingStations[$departureAirportID];

                if ( ($airlineService->service_timings && !$airlineService->service->turnaround_service)
                    || (!$airlineService->service_timings && !$airlineService->turnaround_service)
                ){
                    return null;
                }

                $depArrTime = getFlightDepartureInitialDate($flight);
                break;

            default:
                break;
        }

        if (!$station || !checkSLAEffectivePeriod($airlineService, $depArrTime)) {
            return null;
        }

        $eligibleStaff = StaffService::getStaffWithLocation($airline->id, $airlineService, $station->id);

        /**
         * Prepared Data
         */
        if ($airlineService->service_timings == 1) {
            if ($airlineService->service[$serviceType] != 1) {
                return null;
            }

            $obj = $airlineService->service;
        }
        else {
            if ($airlineService[$serviceType] != 1) {
                return null;
            }

            $obj = $airlineService;
        }

        $req = $arr = $plannedReport = $plannedRelease = $duration = null;

        switch ($serviceType){
            case ARRIVAL_SERVICE:
                $req = "arr_staff_req";
                $arr = getFlightArrivalInitialDatePTA($flight);
                $plannedReport = date("Y-m-d H:i:s", strtotime($arr) - (60 * $obj->arr_report_time));
                $plannedRelease = date("Y-m-d H:i:s", strtotime($arr) + (60 * $obj->arr_release_time));
                $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                break;

            case DEPARTURE_SERVICE:
                $req = "dep_staff_req";
                $dep = getFlightDepartureInitialDatePTD($flight);
                $plannedReport = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_report_time));
                $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_release_time));
                $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                break;

            case TURNAROUND_SERVICE:
                $req = "dep_staff_req";

                $arr = getFlightArrivalInitialDatePTA(getParentFlight($flight));
                $dep = getFlightDepartureInitialDatePTD($flight);

                $plannedReport = date("Y-m-d H:i", strtotime($arr) - (60 * $obj->turn_report_time));
                $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->turn_release_time));

                $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                break;
        }

//        debug("{$serviceType}  - REPORT: {$plannedReport} - {$plannedRelease} : RELEASE");

        $airlineService->service_type = $serviceType;
        $airlineService->sla = ($obj->service ? $obj->service->abbr : $obj->abbr);
        $airlineService->req = $airlineService->{$req};
        $airlineService->report_time = $plannedReport;
        $airlineService->release_time = $plannedRelease;
        $airlineService->duration = $duration;

        $airlineService->eligibleStaff = $eligibleStaff;

        return $airlineService;
    }

}