<?php


namespace App\Classes\FlightStaff;

use App\Models\AirlineService;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\FlightStaff;
use App\Models\StaffService;

class StaffFilter
{
    protected $flight;
    protected $flightNumber;
    protected $airline;
    protected $airport;
    protected $airportID;
    protected $departureAirportID;
    protected $arrivalAirportID;
    protected $serviceType;
    protected $loadAllStationsStaff;

    protected $handlingStations;
    protected $occStaffOnly;

    // OUTPUTS
    protected $eligibleServices = [];

    public function __construct($flight, $loadAllStationsStaff = false, $occStaffOnly = false)
    {
        $this->flight = $flight;
        $this->flightNumber = $flight->flightNumber;
        $this->airline = $flight->flightNumber->airline;
        $this->occStaffOnly = $occStaffOnly;
        $this->loadAllStationsStaff = $loadAllStationsStaff;
        $this->handlingStations = Airport::getHandlingStationsIdAndObject();
        $this->serviceType = $this->getServiceType();
    }

    function getServiceType(){

        $this->departureAirportID = $this->flight->departure_airport_id ? $this->flight->departure_airport_id : $this->flightNumber->departure_airport_id;
        $this->arrivalAirportID = $this->flight->arrival_airport_id ? $this->flight->arrival_airport_id : $this->flightNumber->arrival_airport_id;

        if (array_key_exists($this->departureAirportID, $this->handlingStations)
            && array_key_exists($this->arrivalAirportID, $this->handlingStations))
        {
            return FULL_SERVICE;
        }
        elseif (array_key_exists($this->departureAirportID, $this->handlingStations))
        {
            return DEPARTURE_SERVICE;
        }
        else if (array_key_exists($this->arrivalAirportID, $this->handlingStations))
        {
            return ARRIVAL_SERVICE;
        }

        return null;
    }

    function getFlightWatchStaffData(){

        switch ($this->serviceType){
            case FULL_SERVICE:
                // DEP
                $depArrTime = $this->getDepartureEligibleServices();

                // ARR
                $this->getArrivalEligibleServices();

                $typeText = "DEP-ARR";
                $flightType = "<span class='label label-warning'>DEP-ARR</span>";
                break;

            case DEPARTURE_SERVICE:
                $depArrTime = $this->getDepartureEligibleServices();
                $typeText = "DEP";
                $flightType = "<span class='label label-success'>DEP</span>";
                break;

            case ARRIVAL_SERVICE:
                $depArrTime = $this->getArrivalEligibleServices();
                $typeText = "ARR";
                $flightType = "<span class='label label-primary'>ARR</span>";
                break;

            default:
                $depArrTime = null;
                $typeText = "-";
                $flightType = "-";
                break;
        }

//        debug("SERVICES:");
//        debug($this->eligibleServices);
        $sector = getFlightSector($this->flight);

        return [
            'id'            => $this->flight->id,
            'flight'        => "/{$typeText}/ ".$this->airline->iata."-".$this->flightNumber->flight_number."(".$sector.") ".baseDateFormat($depArrTime, true, "-", true),
            'serviceType'   => $this->serviceType,
            'flightType'    => $flightType,
            "depArrTime"    => $depArrTime,
            'services'      => $this->eligibleServices,
            'station'       => "",
        ];
    }

    protected function getDepartureEligibleServices(){
        $airlineServices = AirlineService::getServices($this->airline->id, $this->departureAirportID, $this->occStaffOnly);
        $depArrTime = getFlightDepartureInitialDate($this->flight);
        $this->getEligibleServices($this->departureAirportID, $airlineServices, DEPARTURE_SERVICE, $depArrTime);

        return $depArrTime;
    }

    protected function getArrivalEligibleServices(){
        $airlineServices = AirlineService::getServices($this->airline->id, $this->arrivalAirportID, $this->occStaffOnly);
        $depArrTime = getFlightArrivalInitialDate($this->flight);
        $this->getEligibleServices($this->arrivalAirportID, $airlineServices, ARRIVAL_SERVICE, $depArrTime);

        return $depArrTime;
    }

    protected function getEligibleServices($stationID, $airlineServices, $serviceType, $depArrTime){
        $rosteredStaffUserIDs = null;

        foreach ($airlineServices as $j => $airlineService) {

            if (!checkSLAEffectivePeriod($airlineService, $depArrTime)){
                continue;
            }

            // SKIP if no appropriate service found
            if ($serviceType == ARRIVAL_SERVICE){
                $service_type = ARRIVAL_SERVICE;

                if ( ($airlineService->service_timings && !$airlineService->service->arrival_service)
                    || (!$airlineService->service_timings && !$airlineService->arrival_service)
                ){
                    continue;
                }
            }
            else {
                $service_type = DEPARTURE_SERVICE;
                if ( ($airlineService->service_timings && !$airlineService->service->departure_service)
                    || (!$airlineService->service_timings && !$airlineService->departure_service)
                ){
                    if (($airlineService->service_timings && !$airlineService->service->turnaround_service)
                        || (!$airlineService->service_timings && !$airlineService->turnaround_service)){
                        continue;
                    }
                    else {
                        if (staffFlightIsInactiveByID($this->flight->parent_id)){
                            continue;
                        }
                        else {
                            $service_type = TURNAROUND_SERVICE;
                        }
                    }
                }
            }
            // End

            if ($this->loadAllStationsStaff){
                $eligibleStaff = StaffService::getStaffWithLocation($this->airline->id, $airlineService, $stationID);
            }
            else {
                $eligibleStaff = StaffService::getStaff($stationID, $this->airline->id, $airlineService);
            }


            /**
             * Prepared Data
             */
            if ($airlineService->service_timings == 1) {
                if ($airlineService->service[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService->service;
            }
            else {
                if ($airlineService[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService;
            }

            switch ($service_type){
                case ARRIVAL_SERVICE:
                    $itemServiceType = ARRIVAL_SERVICE;
                    $req = "arr_staff_req";
                    $min = "arr_staff_min";
                    $arr = getFlightArrivalInitialDatePTA($this->flight);
                    $plannedReport = date("Y-m-d H:i:s", strtotime($arr) - (60 * $obj->arr_report_time));
                    $plannedRelease = date("Y-m-d H:i:s", strtotime($arr) + (60 * $obj->arr_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case DEPARTURE_SERVICE:
                    $itemServiceType = DEPARTURE_SERVICE;
                    $req = "dep_staff_req";
                    $min = "dep_staff_min";
                    $dep = getFlightDepartureInitialDatePTD($this->flight);
                    $plannedReport = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case TURNAROUND_SERVICE:
                    $itemServiceType = TURNAROUND_SERVICE;
                    $req = "turn_staff_req";
                    $min = "turn_staff_min";

                    $arr = getFlightArrivalInitialDatePTA(getParentFlight($this->flight));
                    $dep = getFlightDepartureInitialDatePTD($this->flight);

                    $plannedReport = date("Y-m-d H:i", strtotime($arr) - (60 * $obj->turn_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->turn_release_time));
//                    debug($plannedReport."-".$plannedRelease);
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;
            }

            $airlineServices[$j]->sla = $obj->service ? $obj->service->abbr : $obj->abbr;
            $airlineServices[$j]->req = $airlineService->{$req} ? $airlineService->{$req} : $airlineService->service->{$req};
            $airlineServices[$j]->report_time = $plannedReport;
            $airlineServices[$j]->release_time = $plannedRelease;
            $airlineServices[$j]->duration = $duration;
            $airlineServices[$j]->rosteredStaff = null;
            $airlineServices[$j]->eligibleStaff = $eligibleStaff;
            $airlineServices[$j]->serviceType = $itemServiceType;

            // Add to Services Array
            $this->eligibleServices[] = $airlineServices[$j];
        }
    }
}