<?php
/**
 * Created by PhpStorm.
 * User: dimmer
 */

namespace App\Classes\Crew;


use App\Models\Location;
use App\Repositories\Interfaces\ILocationRepository;

class FDP
{
    // law by which crew is rostered
    protected $rosterLaw = TAJIKISTAN_CAA;

    // Local Time +/- FROM GMT (in secs)
    protected $localTime = 18000;

    // First Flight
    protected $firstFlight;

    // Last Flight
    protected $lastFlight;

    // current calculated report time
    protected $reportTime;

    // current calculated release time
    protected $releaseTime;

    // + minutes which is added To Flights std (default = 1H:15M before first flight)
    protected $reportTimeOffset = 4500;

    // + minutes which is added To Flights sta (default = 0H:30M after last flight)
    protected $releaseTimeOffset = 1800;

    protected $standbyFlightOffset = 1800;

    // Min Rest Time is 12 Hours
    protected $minRestTime = 36000;

    // max fdp time
    protected $maxFdpTime;

    // allowed extended fdp time
    protected $extendedFdpTime;

    // current calculated fdp time
    protected $currentFdpTime;

    protected $flightTime = 0;

    protected $flyingTime = 0;

    protected $otherTime = 0;

    protected $crewBaseAirports = [];

    protected $lastDestinationAirport;

    // list of flights in the current fdp
    protected $flights = [];

    // is class of 1 of 3 types (Regular/DHC/Standby)
    protected $newFlight;

    //
    protected $flight;

    protected $orderFlights;
    /**
     * @param $newFlight
     * @param null $reportTimeOffset
     * @param null $releaseTimeOffset
     * @param array $crewBaseAirports
     * @param string $orderFlights
     */
    public function __construct($newFlight, $reportTimeOffset = null, $releaseTimeOffset = null, $crewBaseAirports = [], $orderFlights = ORDER_DESC){

        if ($reportTimeOffset)
            $this->reportTimeOffset = $reportTimeOffset;

        if ($releaseTimeOffset)
            $this->releaseTimeOffset = $releaseTimeOffset;

        $this->orderFlights = $orderFlights;

        $this->checkAndAddFlight($newFlight);

        $this->setCrewBaseAirports($crewBaseAirports);
    }

    public function getFlights(){
        return array_reverse($this->flights);
        return $this->flights;
    }

    public function getFirstFlight(){
        return $this->firstFlight;
    }

    public function getLastFlight(){
        return $this->lastFlight;
    }

    public function getReportTime(){
        return $this->reportTime;
    }

    public function getReleaseTime(){
        return $this->releaseTime;
    }

    public function getFlightTime(){
        return $this->flightTime;
    }

    public function getFlyingTime(){
        return $this->flyingTime;
    }

    public function getOtherTime(){
        return $this->otherTime;
    }

    public function getCurrentFdpTime(){
        return $this->currentFdpTime;
    }

    public function getMaxFdpTime(){
        return $this->maxFdpTime;
    }

    /**
     * Check If Flight Belongs TO This FDP => true : Add Flight
     * @param $newFlight
     * @return bool
     */
    public function checkAndAddFlight($newFlight){

        $this->newFlight = $newFlight;

        if (count($this->flights)){
            // Set Last Destination Airport Code
            $this->setLastDestinationAirport();

            // Calculate Min Rest Time
            $this->calculateMinRestTime();

            // Check For Duplicates in Flights
            if ($this->checkIfFlightDuplicated())
                return false;

            // If Flight Is Not Within FDP Break And Return False
            if ($this->checkIfFlightNotWithinFdp())
                return false;
        }

        $this->addInOrderedPosition($this->orderFlights);

        $this->setFirstFlightAndReportTime();

        $this->setLastFlightAndReleaseTime();

        $this->calculateAndSetCurrentFdpTime();

        $this->calculateAndSetMaxFdp(TRUE);

        $this->calculatedAndSetFlightTime();

        $this->calculatedAndSetFlyingTime();

        $this->calculatedAndSetOtherTime();

        $this->unsetNewFlight();

        return true;
    }

    public function checkIfFlightDuplicated(){
        foreach ($this->flights as $flight) {
            if ($this->newFlight->getFlight()->id == $flight->id)
                return true;
        }
        return false;
    }

    public function unsetNewFlight(){
        // Unset NewFlight
        $this->newFlight = null;
        // Unset Flight
        $this->flight = null;
    }

    /**
     * Add Given Flight To The FDP
     */
    public function addFlight(){
        $this->flight = $this->newFlight->getFlight();

        $this->flights[] = $this->flight;
    }



    public function addInOrderedPosition($order = ORDER_DESC){
        $this->flight = $this->newFlight->getFlight();
        $flight = $this->flight;

        if (count($this->flights)){
            foreach ($this->flights as $i => $each) {
                switch($order){
                    case ORDER_DESC:
                        if (strtotime($flight->std) > strtotime($each->std)) {
                            array_splice($this->flights, $i, 0, [$flight]);
                            return true;
                        }
                        break;
                    case ORDER_ASC:
                        if (strtotime($flight->std) < strtotime($each->std)) {
                            array_splice($this->flights, $i, 0, [$flight]);
                            return true;
                        }
                        break;
                }
            }
        }
        $this->flights[] = $flight;
        return true;
    }

    public function calculatedAndSetFlightTime(){
        $this->flightTime += $this->newFlight->getFlightTime();
    }

    public function calculatedAndSetFlyingTime(){
        $this->flyingTime += $this->newFlight->getFlyingTime();
    }

    public function calculatedAndSetOtherTime(){
        $this->otherTime = $this->currentFdpTime - $this->flightTime;
    }

    /**
     * Calculate And Set Report Time Out of Flights' List, Set First Flight
     * @return bool
     */
    public function setFirstFlightAndReportTime(){
        if ($this->firstFlight) {
            if ($this->newFlight->checkIfBeforeFlight($this->firstFlight)) {
                // Set Other Times To 0
                //$this->updateAllFlightsOtherTimes(0);
                $this->firstFlight = $this->flight;
            }
        }
        else {
            $this->firstFlight = $this->flight;
        }
        $this->setReportTime();
    }


    public function updateAllFlightsOtherTimes($hours = 0){
        foreach ($this->flights as $flight) {
            $flight->setOtherTime($hours);
        }
    }

    /**
     * Calculate And Set Release Time Out of Flights' List, Set Last Flight
     * @return bool
     */
    public function setLastFlightAndReleaseTime(){
        if ($this->lastFlight) {
            if ($this->newFlight->checkIfAfterFlight($this->lastFlight)) {
                $this->lastFlight = $this->flight;
            }
        }
        else {
            $this->lastFlight = $this->flight;
        }

        $this->setReleaseTime();
    }

    /**
     * Check If Flight Belongs TO This FDP
     * @return bool
     */
    function checkIfFlightNotWithinFdp(){
        $departure = $this->newFlight->getDeparture();
        $arrival = $this->newFlight->getArrival();

        if ($this->checkIfMinRestTimeApplied($departure, $arrival)) {
            return true;
        }
        return false;

        if ($arrival < strtotime($this->firstFlight->std)){
            if ($this->checkIfMinRestTimeApplied($departure, $arrival)) {
                return true;
            }
        }
        else {
            if ($this->lastFlight->is_standby){
                if (strtotime($departure) > strtotime($this->lastFlight->endDuty)) {
                    // If Min Rest Time Applied => It's Another FDP, thus Doesn't Belong To The Current FDP
                    if ($this->checkIfMinRestTimeApplied($departure, $arrival)) {
                        return true;
                    }
                }
            }
            else {
                if (strtotime($departure) > strtotime($this->lastFlight->sta)) {
                    // If Min Rest Time Applied => It's Another FDP, thus Doesn't Belong To The Current FDP
                    if ($this->checkIfMinRestTimeApplied($departure, $arrival)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Check If Between Given Flight And Other FLights of The FDP Min REST Exists
     * @param $departure
     * @param $arrival
     * @return bool
     */
    function checkIfMinRestTimeApplied($departure, $arrival){
        if(strtotime($arrival) < strtotime($this->firstFlight->std)) {
            if (strtotime($this->firstFlight->std) - strtotime($arrival) >=  $this->minRestTime){
                return true;
            }
            else {
                return false;
            }
        }
        else {
            if ($this->lastFlight->is_standby){
                $lastFlightEndDuty = strtotime($this->lastFlight->std) + $this->standbyFlightOffset;
                debug("New Flight Departure: ".$departure." Last Flight End Duty: ". date("Y-m-d H:i", $lastFlightEndDuty));

                if (strtotime($departure) > $lastFlightEndDuty) {
                    if (strtotime($departure) - $lastFlightEndDuty >= $this->minRestTime) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
            else {
                if (strtotime($departure) > strtotime($this->lastFlight->sta)) {
                    if (strtotime($departure) - strtotime($this->lastFlight->sta) >= $this->minRestTime) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        }

    }

    /**
     * Calculate And Set FDP Currents
     */
    function calculateAndSetCurrentFdpTime(){
        if ($this->reportTime && $this->releaseTime){
            $this->currentFdpTime = round((strtotime($this->releaseTime) - strtotime($this->reportTime)) / 3600, 4);
        }
    }

    /**
     * Debug All Important Parameters For Testing
     */
    public function debugDetails(){
        $string = new \stdClass();
        $string->reportTime = $this->reportTime;
        $string->releaseTime = $this->releaseTime;
        $string->maxFdpTime = $this->maxFdpTime;
        $string->currentFdpTime = $this->currentFdpTime;
        $string->flightTime = $this->flightTime;
        $string->flyingTime = $this->flyingTime;
        $string->otherTime = $this->otherTime;
        $string->minRest    = $this->minRestTime;
        $string->flights    = count($this->flights);
        $string->flightsArray = [];
        foreach ($this->flights as $flight) {
            $string->flightsArray[] = "STD: {$flight->std} / STA : {$flight->sta}";
        }
        $string->firstFlight = "STD: {$this->firstFlight->std} / STA : {$this->firstFlight->sta}";
        $string->lastFlight = "STD: {$this->lastFlight->std} / STA : {$this->lastFlight->sta}";
//        $string->lastFlight = $this->lastFlight;
        debug($string);
    }


    /**
     * Calculate Min Rest Time Based On Given Roster Law
     */
    function calculateMinRestTime(){
        // Last Destination Airport
        switch($this->rosterLaw){
            case TAJIKISTAN_CAA:
                if (in_array($this->lastDestinationAirport, $this->crewBaseAirports)) {
                    if ($this->currentFdpTime)
                        $this->minRestTime = max($this->currentFdpTime, 12 * 60 *60);
                    else
                        $this->minRestTime = 12 * 60 * 60;
                }
                else {
                    if ($this->currentFdpTime)
                        $this->minRestTime = max($this->currentFdpTime, 10 * 60 * 60);
                    else
                        $this->minRestTime = 10 * 60 * 60;
                }
                break;
            DEFAULT:
                break;
        }
    }

    function setLastDestinationAirport(){
        if ($this->lastFlight){
            //$this->lastDestinationAirport = $this->getLastFlight()->flightNumber->arrivalAirport->iata;
        }
    }

    function setCrewBaseAirports($airports){
        $this->crewBaseAirports = $airports;
    }

    /**
     * Set Report Time Offset
     * @param $reportTimeOffset
     */
    public function setReportTimeOffset($reportTimeOffset){
        $this->reportTimeOffset = $reportTimeOffset;
    }

    /**
     * Set Release Time Offset
     * @param $releaseTimeOffset
     */
    public function setReleaseOffset($releaseTimeOffset){
        $this->releaseTimeOffset = $releaseTimeOffset;
    }

    /**
     * Set Report Time If Given Time (Default: Calculate And Set Out of First Flight
     * @param null $reportTime
     * @return bool
     */
    public function setReportTime($reportTime = null){
        if ($reportTime) {
            $this->reportTime = $reportTime;
        }
        else {
            if ($this->firstFlight && $this->firstFlight->id == $this->flight->id){
                // If First Flight => is Current Added Flight => Apply Flight Type Report Time Method
                $this->reportTime = $this->newFlight->getReportTime($this->reportTimeOffset);
            }
            else {
                return false;
            }
        }
        return true;
    }

    /**
     * Set Release Time If Given Time (Default: Calculate And Set Out of Last Flight)
     * @param null $releaseTime
     * @return bool
     */
    public function setReleaseTime($releaseTime = null){
        if ($releaseTime) {
            $this->$releaseTime = $releaseTime;
        }
        else {
            if ($this->lastFlight && $this->lastFlight->id == $this->flight->id){
                // If Last Flight => is Current Added Flight => Apply Flight Type Release Time Method
                $this->releaseTime = $this->newFlight->getReleaseTime($this->releaseTimeOffset);
            }
            else {
                return false;
            }
        }
        return true;
    }


    /**
     * Find Max Daily FDP Based on Report time & Number of Sectors
     * @param bool|false $setMaxFdp
     * @return float
     */
    public function calculateAndSetMaxFdp($setMaxFdp = false){
        $hour_minute_format = strtotime(date('H:i', strtotime($this->firstFlight) + $this->localTime));

        $sector_index = count($this->flights) - 1;

        $max_fdp_selected = 12;

        switch($this->rosterLaw) {
            case TAJIKISTAN_CAA:
                $max_fdp = [
                    0 => [12.00, 12.00, 10.50, 10.50, 08.50, 08.50, 08.50, 08.50, 08.50, 08.50, 08.50],
                    1 => [11.00, 11.00, 10.00, 10.00, 06.50, 06.50, 06.50, 06.50, 06.50, 06.50, 06.50],
                ];

                $sector_index = $sector_index > 10 ? 10 : $sector_index;

                if ($hour_minute_format >= strtotime('06:01') && $hour_minute_format <= strtotime('21:59'))
                    $max_fdp_selected = $max_fdp[0][$sector_index]; //index -> sectors
                else
                    $max_fdp_selected = $max_fdp[1][$sector_index]; //index -> sectors
                break;
            case EASA:
                $max_fdp = [
                    0 => [13.00, 13.00, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00],
                    1 => [12.75, 12.75, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00],
                    2 => [12.50, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00],
                    3 => [12.25, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00],
                    4 => [12.00, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00],
                    5 => [11.75, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00, 09.00],
                    6 => [11.50, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00, 09.00],
                    7 => [11.25, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00, 09.00, 09.00],
                    8 => [11.00, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00, 09.00, 09.00],
                    9 => [12.00, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00],
                    10 => [12.25, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00],
                    11 => [12.50, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00],
                    12 => [12.75, 12.75, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.00, 09.00, 09.00],
                ];

                if ($hour_minute_format >= strtotime('06:00') && $hour_minute_format <= strtotime('13:29'))
                    $max_fdp_selected = $max_fdp[0][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('13:30') && $hour_minute_format <= strtotime('13:59'))
                    $max_fdp_selected = $max_fdp[1][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('14:00') && $hour_minute_format <= strtotime('14:29'))
                    $max_fdp_selected = $max_fdp[2][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('14:30') && $hour_minute_format <= strtotime('14:59'))
                    $max_fdp_selected = $max_fdp[3][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('15:00') && $hour_minute_format <= strtotime('15:29'))
                    $max_fdp_selected = $max_fdp[4][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('15:30') && $hour_minute_format <= strtotime('15:59'))
                    $max_fdp_selected = $max_fdp[5][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('16:00') && $hour_minute_format <= strtotime('16:29'))
                    $max_fdp_selected = $max_fdp[6][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('16:30') && $hour_minute_format <= strtotime('16:59'))
                    $max_fdp_selected = $max_fdp[7][$sector_index]; //index -> sectors

                else if ($hour_minute_format >= strtotime('05:00') && $hour_minute_format <= strtotime('05:14'))
                    $max_fdp_selected = $max_fdp[9][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:15') && $hour_minute_format <= strtotime('05:29'))
                    $max_fdp_selected = $max_fdp[10][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:30') && $hour_minute_format <= strtotime('05:44'))
                    $max_fdp_selected = $max_fdp[11][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:45') && $hour_minute_format <= strtotime('05:59'))
                    $max_fdp_selected = $max_fdp[12][$sector_index]; //index -> sectors
                else if (($hour_minute_format >= strtotime('00:00') && $hour_minute_format <= strtotime('04:59')) ||
                    ($hour_minute_format >= strtotime('17:00') && $hour_minute_format <= strtotime('23:59')) ) {
                    $max_fdp_selected = $max_fdp[8][$sector_index]; //index -> sectors
                }
                else
                    $max_fdp_selected = 13; //index -> sectors

                break;
        }

        if ($setMaxFdp){
            $this->maxFdpTime = $max_fdp_selected;
            return true;
        }
        else {
            return floatval($max_fdp_selected);
        }
    }
}
